import ZipJS from 'zip.js';
import { posix as path } from 'path';
import _ from 'lodash';
import Debug from 'debug';
import { createDeferred } from '@signageos/lib/dist/Promise/deferred';

const debug = Debug('@signageos/front-display:Archive:extractor');

export async function extractZipFile(
	archiveFileUri: string,
	ensureDirectory: (filePath: string) => Promise<void>,
	createFile: (filePath: string, blob: Blob) => Promise<void>,
) {
	const response = await window.fetch(archiveFileUri);
	const archiveBlob = await response.blob();
	return await extractZipBlob(archiveBlob, ensureDirectory, createFile);
}

export async function extractZipBlob(
	archiveBlob: Blob,
	ensureDirectory: (filePath: string) => Promise<void>,
	createFile: (filePath: string, blob: Blob) => Promise<void>,
) {
	const { promise, reject } = createDeferred<never>(true);

	return Promise.race([
		promise,
		(async () => {
			const reader = await createZipBlobReader(archiveBlob, reject);
			const entries = await getZipReaderEntries(reader);
			const directories = entries.filter((entry: ZipJS.Entry) => entry.directory).map((entry: ZipJS.Entry) => entry.filename);
			// Based on dependencies of parent directories, create directories in order, parents first.
			const groupedDirectories = groupDirectoriesByParents(directories);
			const fileEntries = entries.filter((entry: ZipJS.Entry) => !entry.directory);

			// Explicit specification of directories in zip file is optional. First try to create defined dirs
			for (const directoryPaths of groupedDirectories) {
				await Promise.all(directoryPaths.map((directoryPath) => ensureDirectory(directoryPath)));
			}

			await Promise.all(
				fileEntries.map(async (entry: ZipJS.Entry) => {
					const blob = await getZipEntryData(entry);
					const parentDirectoryPath = path.dirname(entry.filename);
					if (parentDirectoryPath !== '.') {
						// Explicit specification of directories in zip file is optional. Ensure file dir always before create file
						await ensureDirectory(parentDirectoryPath);
					}
					await createFile(entry.filename, blob);
				}),
			);
			reader.close();
		})(),
	]);
}

export function groupDirectoriesByParents(inputDirs: string[]) {
	const sortedDirs = [...inputDirs].sort((a: string, b: string) => a.split('/').length - b.split('/').length);
	const groupedDirectories = sortedDirs.reduce<string[][]>((groupedDirs: string[][], dir: string) => {
		const parentDir = path.dirname(dir);
		const index = groupedDirs.findIndex((oneGroupDirs: string[]) => oneGroupDirs.includes(parentDir));
		const defaultOneGroupDirs: string[] = [];
		return _.set(groupedDirs, index + 1, _.get(groupedDirs, index + 1, defaultOneGroupDirs).concat(dir));
	}, []);
	return groupedDirectories;
}

export async function getZipExtractSize(blobFile: Blob): Promise<number> {
	const { promise, reject } = createDeferred<never>(true);

	return Promise.race([
		promise,
		(async () => {
			const reader = await createZipBlobReader(blobFile, reject);
			const entries = await getZipReaderEntries(reader);
			const extractSize: number = entries.reduce((usage: number, entry: ZipJS.Entry) => {
				return usage + entry.uncompressedSize;
			}, 0);
			reader.close();
			return extractSize;
		})(),
	]);
}

function createZipBlobReader(archiveBlob: Blob, onError: (e: unknown) => void) {
	return new Promise<ZipJS.ZipReader>((resolve) => ZipJS.createReader(new ZipJS.BlobReader(archiveBlob), resolve, onError));
}

function getZipReaderEntries(reader: ZipJS.ZipReader) {
	return new Promise((resolve: (entries: ZipJS.Entry[]) => void) => reader.getEntries(resolve));
}

function getZipEntryData(entry: ZipJS.Entry) {
	return new Promise((resolve: (blob: Blob) => void) => {
		entry.getData(
			new ZipJS.BlobWriter(''), // TODO content-type
			resolve,
			(current: number, total: number) => debug(`Archive extract progress ${current}/${total}`),
		);
	});
}
