import { Reducer, Store, applyMiddleware, createStore } from 'redux';
import createSagaMiddleware from 'redux-saga';
import { StopApplication } from './applicationActions';
import IBasicDriver from '../NativeDevice/IBasicDriver';
import { EventEmitter } from 'events';
import { actionEmitterMiddleware, createActionForwardMiddleware } from './applicationMiddlewares';

function initializeRedux<IState>(
	reducer: Reducer<IState>,
	getSaga: (store: Store<IState>) => Iterator<any>,
	onError: (e: Error) => void = (error: Error) => console.error(error),
	storeEmitter: EventEmitter,
) {
	const sagaMiddleware = createSagaMiddleware({ onError });
	const store = createStore<IState>(
		reducer,
		applyMiddleware(actionEmitterMiddleware, sagaMiddleware, createActionForwardMiddleware(storeEmitter)),
	);

	if (globalThis.__POSTPONE_SAGA_EXECUTION) {
		globalThis.__runSagas ??= [];
		globalThis.__runSagas.push(() => void sagaMiddleware.run(() => getSaga(store)));
	} else {
		sagaMiddleware.run(() => getSaga(store));
	}

	return store;
}

export async function createApplication<IState>(
	reducer: Reducer<IState>,
	getSaga: (store: Store<IState>) => Iterator<any>,
	getNativeDriver: () => IBasicDriver,
	onError?: (e: Error) => void,
) {
	const storeEmitter = new EventEmitter();
	const nativeDriver = getNativeDriver();
	const store = initializeRedux(reducer, getSaga, onError, storeEmitter);

	nativeDriver.start();

	return {
		store,
		storeEmitter,
		onUnload: () => {
			store.dispatch({ type: StopApplication });
			nativeDriver.stop();
		},
	};
}
