"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.shouldBeOnByTimers = shouldBeOnByTimers;
exports.shouldBeOnByTimerSettings = shouldBeOnByTimerSettings;
exports.orderTimerEventsByChronology = orderTimerEventsByChronology;
exports.getTimerEventsFromShortWeekday = getTimerEventsFromShortWeekday;
exports.getTimerEvents = getTimerEvents;
exports.getNextTimerEventForSameDay = getNextTimerEventForSameDay;
exports.getNextTimerEventForFollowingDays = getNextTimerEventForFollowingDays;
exports.getNextTimerEvent = getNextTimerEvent;
exports.getLastTimerEvent = getLastTimerEvent;
const moment_timezone_1 = __importDefault(require("moment-timezone"));
const converters_1 = require("../converters");
const timeConverter_1 = require("../../DateTime/timeConverter");
const object_1 = require("../../Utils/object");
const TimerEventWeekdayNumber_1 = require("./TimerEventWeekdayNumber");
/**
 * @deprecated use `getLastTimerEvent` and `convertLegacyTimerToShortWeekdayTimer` instead
 */
function shouldBeOnByTimers(timerSettings, now) {
    return shouldBeOnByTimerSettings((0, converters_1.convertLegacyTimerToShortWeekdayTimer)(timerSettings), now);
}
/**
 * @deprecated use `getLastTimerEvent` instead
 */
function shouldBeOnByTimerSettings(timerSettings, now) {
    const lastTimerEvent = getLastTimerEvent(timerSettings, now);
    return lastTimerEvent === null ? null : lastTimerEvent.type === 'ON';
}
function orderTimerEventsByChronology(timerEvents) {
    return timerEvents.sort((a, b) => (a.weekDayNumber - b.weekDayNumber) * (24 * 60 * 60) + // whole day in seconds
        (0, timeConverter_1.timeToSeconds)(a.time) -
        (0, timeConverter_1.timeToSeconds)(b.time));
}
function getTimerEventsFromShortWeekday(allSettings) {
    const timerEvents = [];
    for (const type of (0, object_1.getObjectKeys)(allSettings)) {
        const settings = allSettings[type];
        for (const weekday of settings.weekdays) {
            if (settings.timeOn) {
                timerEvents.push({
                    weekDayNumber: TimerEventWeekdayNumber_1.TimerEventWeekdayNumber[weekday],
                    type: 'ON',
                    time: settings.timeOn,
                    timerType: settings.type,
                });
            }
            if (settings.timeOff) {
                timerEvents.push({
                    weekDayNumber: TimerEventWeekdayNumber_1.TimerEventWeekdayNumber[weekday],
                    type: 'OFF',
                    time: settings.timeOff,
                    timerType: settings.type,
                });
            }
        }
    }
    return timerEvents;
}
function getTimerEvents(allSettings) {
    return getTimerEventsFromShortWeekday((0, converters_1.convertLegacyTimerToShortWeekdayTimer)(allSettings));
}
function getNextTimerEventForSameDay(sortedTimerEvents, now) {
    return sortedTimerEvents.find((timerEvent) => {
        const isThisDay = timerEvent.weekDayNumber === now.getDay();
        const isOnTime = (0, timeConverter_1.timeToSeconds)(timerEvent.time) >= (0, timeConverter_1.timeToSeconds)((0, moment_timezone_1.default)(now).format('HH:mm:ss'));
        return isThisDay && isOnTime;
    });
}
function getNextTimerEventForFollowingDays(sortedTimerEvents, now) {
    return sortedTimerEvents.find((timerEvent) => {
        return timerEvent.weekDayNumber > now.getDay();
    });
}
function getNextTimerEvent(allSettings, now, timerType) {
    const timerEvents = getTimerEventsFromShortWeekday(allSettings);
    // if there is no timer event at all, return null
    if (timerEvents.length === 0) {
        return null;
    }
    const sortedTimerEvents = orderTimerEventsByChronology(timerEvents);
    const filteredSortedTimerEvents = sortedTimerEvents.filter((timerEvent) => {
        return timerEvent.type === timerType;
    });
    // if there is no timer event for the desired timerType, return null
    if (!filteredSortedTimerEvents.length) {
        return null;
    }
    // then we try to find the next timer event on the same day
    const sameDayNextTimerEvent = getNextTimerEventForSameDay(filteredSortedTimerEvents, now);
    if (sameDayNextTimerEvent) {
        return sameDayNextTimerEvent;
    }
    // if there is no timer event on the same day, we are looking for the next timer event
    // for the remaining days of this week
    const followingDaysNextTimerEvent = getNextTimerEventForFollowingDays(filteredSortedTimerEvents, now);
    if (followingDaysNextTimerEvent) {
        return followingDaysNextTimerEvent;
    }
    // above we ensured that:
    // - there is at least one timer event satisfying the desired timerType
    // - there is no timer event for the same day
    // - there is no timer event for the remaining days of this week
    // - therefore, the next expected timer event must be scheduled for the next week
    // expected timers were previously sorted, so we can return the first one
    return filteredSortedTimerEvents[0];
}
function getLastTimerEvent(allSettings, now, timerType) {
    const timerEvents = getTimerEventsFromShortWeekday(allSettings);
    if (timerEvents.length === 0) {
        return null;
    }
    const sortedTimerEvents = orderTimerEventsByChronology(timerEvents);
    const lastTimerEvent = [...sortedTimerEvents].reverse().find((timerEvent) => {
        if (timerType && timerEvent.type !== timerType) {
            return false;
        }
        const inNextDay = timerEvent.weekDayNumber < now.getDay();
        const isThisDay = timerEvent.weekDayNumber === now.getDay();
        const isOnTime = (0, timeConverter_1.timeToSeconds)(timerEvent.time) <= (0, timeConverter_1.timeToSeconds)((0, moment_timezone_1.default)(now).format('HH:mm:ss'));
        return inNextDay || (isThisDay && isOnTime);
    });
    return lastTimerEvent ? lastTimerEvent : [...sortedTimerEvents].reverse()[0];
}
//# sourceMappingURL=powerTimerComputer.js.map