"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDBDriverExporter = void 0;
const prom_client_1 = require("prom-client");
const debug_1 = __importDefault(require("debug"));
const debug = (0, debug_1.default)('@signageos/lib:Metrics:mongoDBDriverExporter');
const defaultOptions = {
    mongodbDriverCommandsSecondsHistogramBuckets: [0.001, 0.005, 0.01, 0.02, 0.03, 0.04, 0.05, 0.1, 0.2, 0.5, 1, 2, 5, 10],
};
/**
 * Exports metrics for the MongoDB driver
 *
 * It has to be a singleton because it registers the metrics in the global registry
 * and it's not possible to register the same metric multiple times.
 */
class MongoDBDriverExporter {
    constructor(register, options) {
        this.register = register;
        this.options = Object.assign(Object.assign({}, defaultOptions), options);
        const connectionPoolLabels = ['name'];
        this.poolSize = new prom_client_1.Gauge({
            name: 'mongodb_driver_pool_size',
            help: 'the current size of the connection pool, including idle and in-use members',
            labelNames: connectionPoolLabels,
            registers: [this.register],
        });
        this.minSize = new prom_client_1.Gauge({
            name: 'mongodb_driver_pool_min',
            help: 'the minimum size of the connection pool',
            labelNames: connectionPoolLabels,
            registers: [this.register],
        });
        this.maxSize = new prom_client_1.Gauge({
            name: 'mongodb_driver_pool_max',
            help: 'the maximum size of the connection pool',
            labelNames: connectionPoolLabels,
            registers: [this.register],
        });
        this.active = new prom_client_1.Gauge({
            name: 'mongodb_driver_pool_active',
            help: 'the count of connections that are currently in use',
            labelNames: connectionPoolLabels,
            registers: [this.register],
        });
        this.wait = new prom_client_1.Gauge({
            name: 'mongodb_driver_pool_wait',
            help: 'the current size of the wait queue for a connection from the pool',
            labelNames: connectionPoolLabels,
            registers: [this.register],
        });
        this.commandsHistogram = new prom_client_1.Histogram({
            name: 'mongodb_driver_commands_seconds',
            help: 'Timer of mongodb commands',
            buckets: this.options.mongodbDriverCommandsSecondsHistogramBuckets,
            labelNames: ['name', 'command'],
            registers: [this.register],
        });
        this.failedCommands = new prom_client_1.Counter({
            name: 'mongodb_driver_commands_failed_total',
            help: 'Counter of failed mongodb commands',
            labelNames: ['name'],
            registers: [this.register],
        });
    }
    /**
     * Start monitoring the given mongo client instance
     *
     * @param mongoClient mongo client instance
     * @param name unique name for the client that will be used as a label to differentiate between multiple clients
     */
    monitorMongoClient(mongoClient, name) {
        const labels = { name };
        mongoClient.on('connectionPoolCreated', (event) => this.onConnectionPoolCreated(labels, event));
        mongoClient.on('connectionPoolClosed', () => this.onConnectionPoolClosed(labels));
        mongoClient.on('connectionCreated', () => this.onConnectionCreated(labels));
        mongoClient.on('connectionClosed', () => this.onConnectionClosed(labels));
        mongoClient.on('connectionCheckOutStarted', () => this.onConnectionCheckOutStarted(labels));
        mongoClient.on('connectionCheckedOut', () => this.onConnectionCheckedOut(labels));
        mongoClient.on('connectionCheckOutFailed', () => this.onConnectionCheckOutFailed(labels));
        mongoClient.on('connectionCheckedIn', () => this.onConnectionCheckedIn(labels));
        mongoClient.on('commandSucceeded', (event) => this.onCommandSucceeded(labels, event));
        mongoClient.on('commandFailed', () => this.onCommandFailed(labels));
    }
    onConnectionPoolCreated(labels, event) {
        this.poolSize.set(labels, 0);
        this.minSize.set(labels, event.options.minPoolSize);
        this.maxSize.set(labels, event.options.maxPoolSize);
        this.active.set(labels, 0);
        this.wait.set(labels, 0);
    }
    onConnectionCreated(labels) {
        this.poolSize.inc(labels);
    }
    onConnectionClosed(labels) {
        this.poolSize.dec(labels);
    }
    onConnectionCheckOutStarted(labels) {
        this.wait.inc(labels);
    }
    onConnectionCheckedOut(labels) {
        this.active.inc(labels);
        this.wait.dec(labels);
    }
    onConnectionCheckOutFailed(labels) {
        debug('connectionCheckOutFailed');
        this.wait.dec(labels);
    }
    onConnectionCheckedIn(labels) {
        this.active.dec(labels);
    }
    onConnectionPoolClosed(labels) {
        debug('connectionPoolClosed');
        this.poolSize.set(labels, 0);
        this.minSize.set(labels, 0);
        this.maxSize.set(labels, 0);
        this.active.set(labels, 0);
        this.wait.set(labels, 0);
    }
    onCommandSucceeded(labels, event) {
        const fullLabels = Object.assign(Object.assign({}, labels), { command: event.commandName });
        this.commandsHistogram.observe(fullLabels, event.duration / 1000);
    }
    onCommandFailed(labels) {
        this.failedCommands.inc(labels);
    }
}
exports.MongoDBDriverExporter = MongoDBDriverExporter;
//# sourceMappingURL=mongoDBDriverExporter.js.map