#!/usr/bin/env node
import semver from 'semver';
import { readJSON } from './utils.js';

(async () => {
	const packagePath = process.cwd() + '/package.json';
	const pkg = await readJSON(packagePath);

	const ignorePath = process.cwd() + '/.check-depsignore';
	let ignore = [];
	try {
		ignore = await readJSON(ignorePath);
		console.log(`Ignoring packages from ${ignorePath}`);
	} catch (_error) {
		console.log('No ignore file found');
	}

	interface CheckDepsRc {
		specials: string[];
		parsers: Record<string, string>;
		detectors: string[];
		'ignore-dirs': string[];
		ignores: string[];
		expectRanges: Record<string, string>;
	}

	const runControlPath = process.cwd() + '/.check-depsrc.json';
	let runControl: CheckDepsRc | undefined;
	try {
		runControl = await readJSON(runControlPath);
		console.log(`Using run control from ${runControlPath}`);
		if (runControl!.ignores) {
			ignore = ignore.concat(runControl!.ignores);
		}
	} catch (error) {
		console.log('No Run Control file found');
	}

	const includeRegExp = new RegExp(process.argv[2] || '.+');
	const excludeRegExp = new RegExp(process.argv[3] || '^$');

	const deps = { ...(pkg.dependencies || {}), ...(pkg.devDependencies || {}) };

	const depNames = Object.keys(deps);

	const matchedDepNames = depNames.filter((depName) => includeRegExp.test(depName) && !excludeRegExp.test(depName));

	const prereleaseDepNames = [];
	const inexactDepNames = [];
	const expectedRangeDepNames = [];

	function isExpectedRangeValid(depName: string, version: string) {
		return version.startsWith(runControl?.expectRanges[depName]!);
	}

	for (const depName of matchedDepNames) {
		if (ignore.includes(depName)) {
			continue;
		}
		const depVersion = deps[depName];
		const minVersion = semver.minVersion(depVersion);
		if (!minVersion) {
			throw new Error(`Invalid depversion ${depVersion}`);
		}
		if (semver.prerelease(minVersion) !== null) {
			prereleaseDepNames.push(depName);
		}
		if (runControl?.expectRanges && depName in runControl.expectRanges) {
			if (!isExpectedRangeValid(depName, depVersion)) {
				expectedRangeDepNames.push(depName);
			}
		} else {
			if (!semver.valid(depVersion)) {
				// Check inexact only when not expected range
				inexactDepNames.push(depName);
			}
		}
	}

	if (prereleaseDepNames.length > 0) {
		const prereleaseDeps: Record<string, string> = {};
		prereleaseDepNames.forEach((depName) => {
			prereleaseDeps[depName] = deps[depName];
		});
		console.error(`Some packages has prerelease tags in version in package.json deps`, prereleaseDepNames.join(', '), prereleaseDeps);
	}

	if (expectedRangeDepNames.length > 0) {
		const expectedRangeDeps: Record<string, string> = {};
		expectedRangeDepNames.forEach((depName) => {
			expectedRangeDeps[depName] = deps[depName];
		});
		console.error(
			`Some packages has expectation of range of versions specified in package.json deps`,
			expectedRangeDepNames.map((depName) => `${depName}->"${runControl?.expectRanges[depName]}"`).join(', '),
			expectedRangeDeps,
		);
	}

	if (inexactDepNames.length > 0) {
		const inexactDeps: Record<string, string> = {};
		inexactDepNames.forEach((depName) => {
			inexactDeps[depName] = deps[depName];
		});
		console.error(`Some packages has not exact version specified in package.json deps`, inexactDepNames.join(', '), inexactDeps);
	}

	if (prereleaseDepNames.length > 0 || expectedRangeDepNames.length > 0 || inexactDepNames.length > 0) {
		process.exit(1);
	}
})();
