"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const events_1 = require("events");
const lockedDecorator_1 = require("@signageos/lib/es6/Lock/lockedDecorator");
const debugDecorator_1 = require("@signageos/lib/es6/Debug/debugDecorator");
const ISynchronizer_1 = require("../ISynchronizer");
const debug_1 = __importDefault(require("debug"));
const DEBUG_NAMESPACE = '@signageos/front-display:Front:Applet:Sync:SocketSynchronizer';
const logDebug = (0, debug_1.default)(DEBUG_NAMESPACE);
const logDebugSocket = (0, debug_1.default)(DEBUG_NAMESPACE + ':Socket');
/**
 * API to synchronize content with other devices in the network.
 * When online
 *     - If everything is fine and device is always online, the logic is simple - initialize group on the server to determine,
 *       which devices belong together and whenever wait() is called, wait until all other devices are waiting too
 *       and resolve it on all simultaniously.
 * When offline
 *     - If the device goes offline in any point in the process, it silently switches to offline mode, which means that
 *       the SocketSynchronizer is itself acting like a mock server and simply validates and confirms all the requests to
 *       the API immediately to maintain smooth and seamless operation.
 *     - SocketSynchronizer also buffers all the initialized groups while offline and when it goes back online, it dumps the buffer
 *       to the server to synchronize the internal state.
 */
class SocketSynchronizer {
    constructor(getNativeDriver, createSocket, defaultServerUri) {
        this.getNativeDriver = getNativeDriver;
        this.createSocket = createSocket;
        this.defaultServerUri = defaultServerUri;
        this.connected = false;
        this.groups = {};
        this.emitter = new events_1.EventEmitter();
    }
    connect(serverUri) {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.socket) {
                yield this.close();
            }
            this.initSocket(serverUri || this.defaultServerUri);
        });
    }
    close() {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.socket) {
                throw new Error("Can't close when not connected");
            }
            this.socket.close();
            this.socket = undefined;
            this.connected = false;
            Object.keys(this.groups).forEach((groupName) => {
                this.emitter.emit(ISynchronizer_1.SynchronizerEvent.GroupLeft, groupName);
            });
            this.groups = {};
        });
    }
    joinGroup({ groupName, deviceIdentification }) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.socket) {
                throw new Error('Not connected. Call connect() first.');
            }
            const deviceId = deviceIdentification !== undefined ? deviceIdentification : Math.random().toString(32);
            if (this.connected) {
                yield this.joinGroupOnline(groupName, deviceId);
            }
            else {
                this.joinGroupOffline(groupName, deviceId);
            }
        });
    }
    leaveGroup(groupName) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.socket) {
                throw new Error('Not connected. Call connect() first.');
            }
            if (!this.groups[groupName]) {
                throw new Error(`Group ${groupName} isn't joined`);
            }
            if (this.connected) {
                yield this.leaveGroupOnline(groupName);
            }
            else {
                this.leaveGroupOffline(groupName);
            }
        });
    }
    getDeviceIdentification(groupName) {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.groups[groupName]) {
                return this.groups[groupName].deviceIdentification;
            }
            else {
                return undefined;
            }
        });
    }
    isConnected() {
        return __awaiter(this, void 0, void 0, function* () {
            return !!this.socket;
        });
    }
    wait({ groupName, data, timeoutMs }) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.socket) {
                throw new Error('Not connected. Call connect() first.');
            }
            if (typeof timeoutMs !== 'undefined' && timeoutMs <= 0) {
                throw new Error('Timeout must be greater than 0');
            }
            if (this.connected) {
                return yield this.callOnlineWait(groupName, data, timeoutMs);
            }
            else {
                return this.callOfflineWait(groupName, data);
            }
        });
    }
    cancelWait(_groupName) {
        return __awaiter(this, void 0, void 0, function* () {
            throw new Error('Not implemented yet');
        });
    }
    broadcastValue({ groupName, key, value }) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.socket) {
                throw new Error('Not connected. Call connect() first.');
            }
            if (!this.groups[groupName]) {
                throw new Error('Group ' + groupName + " wasn't initialized");
            }
            this.socket.emit('request_set_value', { groupName, key, value });
            // if not connected, emit the value back to oneself
            if (!this.connected) {
                this.emitValue(groupName, key, value);
            }
        });
    }
    isMaster(groupName) {
        return __awaiter(this, void 0, void 0, function* () {
            const group = this.groups[groupName];
            if (group && group.lastStatus) {
                return group.lastStatus.isMaster;
            }
            else {
                return true;
            }
        });
    }
    addListener(event, listener) {
        this.emitter.addListener(event, listener);
    }
    removeListener(event, listener) {
        this.emitter.removeListener(event, listener);
    }
    initSocket(serverUri) {
        this.socket = this.createSocket(serverUri);
        this.debugSocket(this.socket, serverUri);
        this.socket.on('connect', () => __awaiter(this, void 0, void 0, function* () {
            // flush all offline groups to the server
            for (let groupName of Object.keys(this.groups)) {
                if (this.groups[groupName].isOnline === false) {
                    try {
                        yield this.joinGroupOnline(groupName, this.groups[groupName].deviceIdentification);
                    }
                    catch (error) {
                        console.error('flush offline initialized group to the server', groupName, error);
                    }
                }
            }
            this.connected = true;
        }));
        this.socket.on('disconnect', () => {
            this.connected = false;
            // when device disconnects, all initialized groups are now offline and has to be flushed to the server on reconnection
            for (let groupName of Object.keys(this.groups)) {
                this.groups[groupName].isOnline = false;
                this.groups[groupName].lastStatus = undefined;
            }
        });
        this.socket.on('set_value', (groupName, key, value) => {
            if (this.groups[groupName]) {
                const broadcastedValue = { groupName, key, value };
                this.emitter.emit(ISynchronizer_1.SynchronizerEvent.BroadcastedValue, broadcastedValue);
            }
        });
        this.socket.on('device_status', (connectedPeers, groupName) => {
            if (this.groups[groupName]) {
                const master = this.pickMaster(connectedPeers);
                const isMaster = master === this.groups[groupName].deviceIdentification;
                const groupStatus = { groupName, connectedPeers, isMaster };
                this.groups[groupName].lastStatus = groupStatus;
                this.emitter.emit(ISynchronizer_1.SynchronizerEvent.GroupStatus, groupStatus);
            }
        });
    }
    debugSocket(socket, serverUri) {
        logDebugSocket('open socket', { serverUri });
        socket.on('connect', () => logDebugSocket('connect'));
        socket.on('connect_error', (error) => logDebugSocket('connect_error', error));
        socket.on('connect_timeout', () => logDebugSocket('connect_timeout'));
        socket.on('error', (error) => logDebugSocket('error', error));
        socket.on('disconnect', () => logDebugSocket('disconnect'));
        socket.on('reconnect_attempt', () => logDebugSocket('reconnect_attempt'));
        socket.on('reconnecting', () => logDebugSocket('reconnecting'));
        socket.on('reconnect_error', (error) => logDebugSocket('reconnect_error', error));
        socket.on('reconnect_failed', () => logDebugSocket('reconnect_failed'));
        socket.on('ping', () => logDebugSocket('ping'));
        socket.on('pong', () => logDebugSocket('pong'));
    }
    joinGroupOnline(groupName, deviceIdentification) {
        return __awaiter(this, void 0, void 0, function* () {
            const nativeDriver = this.getNativeDriver();
            const deviceUid = yield nativeDriver.getDeviceUid();
            yield new Promise((resolve, reject) => {
                this.socket.emit('join_group', { deviceUid, groupName, deviceIdentification }, (error) => {
                    if (error) {
                        reject(error);
                    }
                    else {
                        resolve();
                    }
                });
            });
            this.groups[groupName] = {
                isOnline: true,
                deviceIdentification,
            };
        });
    }
    joinGroupOffline(groupName, deviceIdentification) {
        this.groups[groupName] = {
            isOnline: false,
            deviceIdentification,
        };
    }
    leaveGroupOnline(groupName) {
        return __awaiter(this, void 0, void 0, function* () {
            yield new Promise((resolve, reject) => {
                this.socket.emit('leave_group', { groupName }, (error) => {
                    if (error) {
                        reject(error);
                    }
                    else {
                        resolve();
                    }
                });
            });
            delete this.groups[groupName];
        });
    }
    leaveGroupOffline(groupName) {
        delete this.groups[groupName];
    }
    callOnlineWait(groupName, data, timeout) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield new Promise((resolve, reject) => {
                let syncConfirmedListener;
                let disconnectListener;
                let timeoutHandler;
                syncConfirmedListener = (confirmedGroupName, confirmData) => {
                    if (confirmedGroupName === groupName) {
                        if (typeof timeoutHandler !== 'undefined') {
                            clearTimeout(timeoutHandler);
                        }
                        this.socket.off('disconnect', disconnectListener);
                        this.socket.off('sync_confirmed', syncConfirmedListener);
                        resolve(confirmData);
                    }
                };
                disconnectListener = () => {
                    if (typeof timeoutHandler !== 'undefined') {
                        clearTimeout(timeoutHandler);
                    }
                    this.socket.off('disconnect', disconnectListener);
                    this.socket.off('sync_confirmed', syncConfirmedListener);
                    // if it goes offline, it should automatically confirm all waits as if nothing happened
                    try {
                        const confirmData = this.callOfflineWait(groupName, data);
                        resolve(confirmData);
                    }
                    catch (error) {
                        reject(error);
                    }
                };
                this.socket.on('sync_confirmed', syncConfirmedListener);
                this.socket.on('disconnect', disconnectListener);
                this.socket.emit('sync_request', { groupName, data }, (error) => {
                    if (error) {
                        this.socket.off('disconnect', disconnectListener);
                        this.socket.off('sync_confirmed', syncConfirmedListener);
                        reject(error);
                    }
                });
                if (typeof timeout !== 'undefined') {
                    timeoutHandler = setTimeout(() => {
                        syncConfirmedListener(groupName, data);
                        this.socket.emit('sync_request_cancel', { groupName });
                    }, timeout);
                }
            });
        });
    }
    callOfflineWait(groupName, data) {
        if (typeof this.groups[groupName] === 'undefined') {
            throw new Error('Group ' + groupName + "wasn't initialized");
        }
        else {
            logDebug('offline wait confirmed for group ' + groupName);
            return data;
        }
    }
    emitValue(groupName, key, value) {
        if (!this.groups[groupName]) {
            throw new Error('Group ' + groupName + "wasn't initialized");
        }
        this.emitter.emit(ISynchronizer_1.SynchronizerEvent.BroadcastedValue, { groupName, key, value });
    }
    /**
     * Picks a master from a list of peers in a deterministic way
     */
    pickMaster(peers) {
        return [...peers].sort()[0];
    }
}
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String]),
    __metadata("design:returntype", Promise)
], SocketSynchronizer.prototype, "connect", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], SocketSynchronizer.prototype, "close", null);
__decorate([
    (0, lockedDecorator_1.locked)('group', { scope: 'instance' }),
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], SocketSynchronizer.prototype, "joinGroup", null);
__decorate([
    (0, lockedDecorator_1.locked)('group', { scope: 'instance' }),
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String]),
    __metadata("design:returntype", Promise)
], SocketSynchronizer.prototype, "leaveGroup", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], SocketSynchronizer.prototype, "wait", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String]),
    __metadata("design:returntype", Promise)
], SocketSynchronizer.prototype, "cancelWait", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], SocketSynchronizer.prototype, "broadcastValue", null);
exports.default = SocketSynchronizer;
//# sourceMappingURL=SocketSynchronizer.js.map