/// <reference types="socket.io-client" />
import ISynchronizer, { BroadcastedValueArgs, BroadcastedValueCallback, ClosedCallback, GroupLeftCallback, GroupStatusCallback, JoinGroupArgs, SynchronizerEvent, WaitArgs } from '../ISynchronizer';
import IDriver from '../../NativeDevice/Front/IFrontDriver';
/**
 * API to synchronize content with other devices in the network.
 * When online
 *     - If everything is fine and device is always online, the logic is simple - initialize group on the server to determine,
 *       which devices belong together and whenever wait() is called, wait until all other devices are waiting too
 *       and resolve it on all simultaniously.
 * When offline
 *     - If the device goes offline in any point in the process, it silently switches to offline mode, which means that
 *       the SocketSynchronizer is itself acting like a mock server and simply validates and confirms all the requests to
 *       the API immediately to maintain smooth and seamless operation.
 *     - SocketSynchronizer also buffers all the initialized groups while offline and when it goes back online, it dumps the buffer
 *       to the server to synchronize the internal state.
 */
declare class SocketSynchronizer implements ISynchronizer {
    private getNativeDriver;
    private createSocket;
    private defaultServerUri;
    private socket;
    private connected;
    private groups;
    private emitter;
    constructor(getNativeDriver: () => IDriver, createSocket: (serverUri: string) => SocketIOClient.Socket, defaultServerUri: string);
    connect(serverUri?: string): Promise<void>;
    close(): Promise<void>;
    joinGroup({ groupName, deviceIdentification }: JoinGroupArgs): Promise<void>;
    leaveGroup(groupName: string): Promise<void>;
    getDeviceIdentification(groupName: string): Promise<string | undefined>;
    isConnected(): Promise<boolean>;
    wait({ groupName, data, timeoutMs }: WaitArgs): Promise<unknown>;
    cancelWait(_groupName: string): Promise<void>;
    broadcastValue({ groupName, key, value }: BroadcastedValueArgs): Promise<void>;
    isMaster(groupName: string): Promise<boolean>;
    addListener(event: SynchronizerEvent.GroupStatus, listener: GroupStatusCallback): void;
    addListener(event: SynchronizerEvent.GroupLeft, listener: GroupLeftCallback): void;
    addListener(event: SynchronizerEvent.BroadcastedValue, listener: BroadcastedValueCallback): void;
    addListener(event: SynchronizerEvent.Closed, listener: ClosedCallback): void;
    removeListener(event: SynchronizerEvent.GroupStatus, listener: GroupStatusCallback): void;
    removeListener(event: SynchronizerEvent.GroupLeft, listener: GroupLeftCallback): void;
    removeListener(event: SynchronizerEvent.BroadcastedValue, listener: BroadcastedValueCallback): void;
    removeListener(event: SynchronizerEvent.Closed, listener: ClosedCallback): void;
    private initSocket;
    private debugSocket;
    private joinGroupOnline;
    private joinGroupOffline;
    private leaveGroupOnline;
    private leaveGroupOffline;
    private callOnlineWait;
    private callOfflineWait;
    private emitValue;
    /**
     * Picks a master from a list of peers in a deterministic way
     */
    private pickMaster;
}
export default SocketSynchronizer;
