"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.P2PSynchronizer = void 0;
const events_1 = require("events");
const debug_1 = __importDefault(require("debug"));
const lockedDecorator_1 = require("@signageos/lib/es6/Lock/lockedDecorator");
const debugDecorator_1 = require("@signageos/lib/dist/Debug/debugDecorator");
const generator_1 = require("@signageos/lib/dist/Hash/generator");
const wait_1 = __importDefault(require("@signageos/lib/dist/Timer/wait"));
const ISynchronizer_1 = require("../ISynchronizer");
const IGroup_1 = require("../Group/IGroup");
const Group_1 = require("../Group/Group");
const WaitService_1 = require("../Group/WaitService");
const IBroadcastService_1 = require("../Group/IBroadcastService");
const BroadcastService_1 = require("../Group/BroadcastService");
const IPeerNetwork_1 = require("../../PeerNetwork/IPeerNetwork");
const DEBUG_NAMESPACE = '@signageos/front-display:Synchronization:P2PSynchronizer';
const logDebug = (0, debug_1.default)(DEBUG_NAMESPACE);
function defaultGroupFactory(args) {
    return new Group_1.Group(args);
}
const defaultOptions = {
    deviceStatusInterval: 30e3,
};
/**
 * Synchronizes devices via provided IPeerDiscoveryService
 *
 * The original purpose of this is to implement synchronization of devices in
 * local network via UDP without a need for an external synchronization server.
 */
class P2PSynchronizer {
    constructor(peerNetwork, options = {}, groupFactory = defaultGroupFactory) {
        this.peerNetwork = peerNetwork;
        this.groupFactory = groupFactory;
        this.groups = {};
        this.emitter = new events_1.EventEmitter();
        this.options = Object.assign(Object.assign({}, defaultOptions), options);
        this.handlePeerNetworkClosed = this.handlePeerNetworkClosed.bind(this);
        this.listenToPeerNetworkEvents();
    }
    connect(_serverUri) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.peerNetwork.start();
        });
    }
    close() {
        return __awaiter(this, void 0, void 0, function* () {
            const groupKeys = Object.keys(this.groups);
            for (let i = 0; i < groupKeys.length; i++) {
                try {
                    const group = this.groups[groupKeys[i]];
                    yield group.leave();
                }
                catch (error) {
                    // errors can only be logged but not thrown because the close has to finish
                    // if group leave fails, it's probably because the peer network is already closed
                    // or something even worse happened that we can't fix
                    console.warn('Failed to leave group', error);
                }
            }
            if (this.peerNetwork.isStarted()) {
                yield this.peerNetwork.stop();
            }
            this.groups = {};
        });
    }
    isConnected() {
        return __awaiter(this, void 0, void 0, function* () {
            return this.peerNetwork.isStarted();
        });
    }
    joinGroup({ groupName, deviceIdentification }) {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.groups[groupName]) {
                console.warn(`Trying to join group ${groupName}, but it's already joined`);
                return;
            }
            const deviceId = deviceIdentification !== null && deviceIdentification !== void 0 ? deviceIdentification : (0, generator_1.generateUniqueHash)(10);
            this.groups[groupName] = yield this.createGroup({ deviceId, groupName });
        });
    }
    leaveGroup(groupName) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.groups[groupName]) {
                console.warn(`Trying to leave group ${groupName} but it's not joined`);
                return;
            }
            yield this.groups[groupName].leave();
            delete this.groups[groupName];
        });
    }
    getDeviceIdentification(groupName) {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.groups[groupName]) {
                return this.groups[groupName].group.getMe().name;
            }
            else {
                return undefined;
            }
        });
    }
    wait({ groupName, data, timeoutMs }) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.groups[groupName]) {
                throw new Error(`Group ${groupName} isn't initialized, call joinGroup() first`);
            }
            if (timeoutMs !== undefined && timeoutMs <= 0) {
                throw new Error('timeout must be greater than 0');
            }
            const waitService = this.groups[groupName].waitService;
            return yield this.waitOrTimeout(waitService, data, timeoutMs);
        });
    }
    cancelWait(groupName) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.groups[groupName]) {
                console.warn(`cancelWait can't be performed. Group ${groupName} isn't initialized, call joinGroup() first`);
                return;
            }
            const waitService = this.groups[groupName].waitService;
            yield waitService.cancelWait();
        });
    }
    broadcastValue({ groupName, key, value }) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.groups[groupName]) {
                throw new Error(`Group ${groupName} isn't initialized, call joinGroup() first`);
            }
            const broadcastService = this.groups[groupName].broadcastService;
            yield broadcastService.broadcastValue(key, value);
        });
    }
    isMaster(groupName) {
        return __awaiter(this, void 0, void 0, function* () {
            const group = this.groups[groupName];
            if (!group) {
                return true;
            }
            const master = group.group.getMaster();
            const me = group.group.getMe();
            return master.id === me.id;
        });
    }
    addListener(event, listener) {
        this.emitter.addListener(event, listener);
    }
    removeListener(event, listener) {
        this.emitter.removeListener(event, listener);
    }
    createGroup({ deviceId, groupName }) {
        return __awaiter(this, void 0, void 0, function* () {
            const group = this.groupFactory({
                name: deviceId,
                groupName,
                peerNetwork: this.peerNetwork,
            });
            yield group.join();
            const waitService = new WaitService_1.WaitService(group);
            yield waitService.start();
            const broadcastService = new BroadcastService_1.BroadcastService(group);
            broadcastService.start();
            const updateStatus = () => __awaiter(this, void 0, void 0, function* () {
                const members = group.getAllMembers();
                const connectedPeers = members.map((member) => member.name);
                const master = group.getMaster();
                const me = group.getMe();
                const isMaster = master.id === me.id;
                const groupStatus = { groupName, connectedPeers, isMaster };
                logDebug('group status changed', groupStatus);
                this.emitter.emit(ISynchronizer_1.SynchronizerEvent.GroupStatus, groupStatus);
            });
            group.addListener(IGroup_1.GroupEvent.MemberJoined, updateStatus);
            group.addListener(IGroup_1.GroupEvent.MemberLeft, updateStatus);
            broadcastService.addListener(IBroadcastService_1.BroadcastEvent.Value, (key, value) => {
                const broadcastedValue = { groupName, key, value };
                logDebug('received broadcasted value', broadcastedValue);
                this.emitter.emit(ISynchronizer_1.SynchronizerEvent.BroadcastedValue, broadcastedValue);
            });
            const interval = setInterval(updateStatus, this.options.deviceStatusInterval);
            const leave = () => __awaiter(this, void 0, void 0, function* () {
                group.removeListener(IGroup_1.GroupEvent.MemberJoined, updateStatus);
                group.removeListener(IGroup_1.GroupEvent.MemberLeft, updateStatus);
                clearInterval(interval);
                waitService.stop();
                broadcastService.stop();
                yield group.leave();
                group.removeAllListeners();
            });
            return {
                group,
                waitService,
                broadcastService,
                leave,
            };
        });
    }
    waitOrTimeout(waitService, data, timeoutMs) {
        return __awaiter(this, void 0, void 0, function* () {
            const waitPromise = waitService.sendWaitMessageAndWaitForOthers(data);
            if (timeoutMs) {
                const timeoutPromise = this.resolveWithDataAfterTimeout(data, timeoutMs);
                try {
                    return yield Promise.race([waitPromise, timeoutPromise]);
                }
                finally {
                    timeoutPromise.cancel();
                }
            }
            else {
                return yield waitPromise;
            }
        });
    }
    resolveWithDataAfterTimeout(data, timeoutMs) {
        const waitPromise = (0, wait_1.default)(timeoutMs);
        const finalPromise = waitPromise.then(() => data);
        finalPromise.cancel = () => waitPromise.cancel();
        return finalPromise;
    }
    listenToPeerNetworkEvents() {
        this.peerNetwork.addListener(IPeerNetwork_1.PeerNetworkEvent.Closed, this.handlePeerNetworkClosed);
    }
    handlePeerNetworkClosed(error) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                yield this.close();
            }
            catch (e) {
                console.warn('Failed to close synchronizer', e);
            }
            this.emitter.emit(ISynchronizer_1.SynchronizerEvent.Closed, error);
        });
    }
}
exports.P2PSynchronizer = P2PSynchronizer;
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    (0, lockedDecorator_1.locked)('sync', { scope: 'instance' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], P2PSynchronizer.prototype, "connect", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    (0, lockedDecorator_1.locked)('sync', { scope: 'instance' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], P2PSynchronizer.prototype, "close", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    (0, lockedDecorator_1.locked)('sync', { scope: 'instance' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], P2PSynchronizer.prototype, "joinGroup", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    (0, lockedDecorator_1.locked)('sync', { scope: 'instance' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String]),
    __metadata("design:returntype", Promise)
], P2PSynchronizer.prototype, "leaveGroup", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], P2PSynchronizer.prototype, "wait", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String]),
    __metadata("design:returntype", Promise)
], P2PSynchronizer.prototype, "cancelWait", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], P2PSynchronizer.prototype, "broadcastValue", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String]),
    __metadata("design:returntype", Promise)
], P2PSynchronizer.prototype, "isMaster", null);
//# sourceMappingURL=P2PSynchronizer.js.map