"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.WaitService = void 0;
const events_1 = require("events");
const debug_1 = __importDefault(require("debug"));
const lockedDecorator_1 = require("@signageos/lib/es6/Lock/lockedDecorator");
const debugDecorator_1 = require("@signageos/lib/es6/Debug/debugDecorator");
const IGroup_1 = require("./IGroup");
const DEBUG_NAMESPACE = '@signageos/front-display:Synchronization:WaitService';
const logDebug = (0, debug_1.default)(DEBUG_NAMESPACE);
var MessageType;
(function (MessageType) {
    MessageType["Query"] = "query";
    MessageType["Wait"] = "wait";
    MessageType["NotWaiting"] = "not_waiting";
})(MessageType || (MessageType = {}));
function isMessage(message) {
    return typeof message === 'object' && message !== null && 'type' in message;
}
function isQueryMessage(message) {
    return isMessage(message) && message.type === MessageType.Query;
}
function isWaitMessage(message) {
    return isMessage(message) && message.type === MessageType.Wait;
}
function isNotWaitingMessage(message) {
    return isMessage(message) && message.type === MessageType.NotWaiting;
}
var Event;
(function (Event) {
    Event["AllPeersWaiting"] = "all_peers_waiting";
    Event["WaitCanceled"] = "wait_canceled";
})(Event || (Event = {}));
/**
 * WaitService handles synchronization of runtime with others in the network via wait interface
 */
class WaitService {
    constructor(group) {
        this.group = group;
        this.started = false;
        this.currentWait = { waiting: false };
        this.waitingPeers = {};
        this.emitter = new events_1.EventEmitter();
        this.handleData = this.handleData.bind(this);
        this.handlePeersChanged = this.handlePeersChanged.bind(this);
    }
    sendWaitMessageAndWaitForOthers(data) {
        return __awaiter(this, void 0, void 0, function* () {
            this.logDebug('sendWaitMessageAndWaitForOthers', data);
            this.currentWait = { waiting: true, data };
            try {
                yield this.sendWaitMessage(data);
            }
            catch (error) {
                this.currentWait = { waiting: false };
                throw error;
            }
            const me = this.group.getMe();
            this.waitingPeers[me.id] = data;
            try {
                return yield this.waitForAllPeers();
            }
            finally {
                this.currentWait = { waiting: false };
            }
        });
    }
    cancelWait() {
        return __awaiter(this, void 0, void 0, function* () {
            this.logDebug('cancelWait');
            this.emitter.emit(Event.WaitCanceled);
            yield this.sendNotWaitingMessage();
            const me = this.group.getMe();
            delete this.waitingPeers[me.id];
        });
    }
    start() {
        return __awaiter(this, void 0, void 0, function* () {
            this.logDebug('start');
            if (this.started) {
                throw new Error('wait service already started');
            }
            yield this.sendQueryMessage();
            this.group.addListener(IGroup_1.GroupEvent.Data, this.handleData);
            this.group.addListener(IGroup_1.GroupEvent.MemberLeft, this.handlePeersChanged);
            this.started = true;
        });
    }
    stop() {
        this.logDebug('stop');
        if (!this.started) {
            throw new Error("wait service isn't started");
        }
        this.group.removeListener(IGroup_1.GroupEvent.Data, this.handleData);
        this.group.removeListener(IGroup_1.GroupEvent.MemberLeft, this.handlePeersChanged);
        this.started = false;
    }
    handleData(data) {
        return __awaiter(this, void 0, void 0, function* () {
            const { from, data: message } = data;
            if (isQueryMessage(message)) {
                yield this.handleQueryMessage(from, message);
            }
            else if (isWaitMessage(message)) {
                this.handleWaitMessage(from, message);
            }
            else if (isNotWaitingMessage(message)) {
                this.handleNotWaitingMessage(from, message);
            }
        });
    }
    handleQueryMessage(from, message) {
        return __awaiter(this, void 0, void 0, function* () {
            this.logDebug('got query message from ' + from.id, JSON.stringify(message));
            if (this.currentWait.waiting) {
                yield this.sendWaitMessage(this.currentWait.data);
            }
        });
    }
    handleWaitMessage(from, message) {
        return __awaiter(this, void 0, void 0, function* () {
            this.logDebug('got wait message from ' + from.id, JSON.stringify(message));
            this.waitingPeers[from.id] = message.data;
            yield this.emitEventIfAllPeersWaiting();
        });
    }
    handleNotWaitingMessage(from, message) {
        this.logDebug('got not waiting message from ' + from.id, JSON.stringify(message));
        delete this.waitingPeers[from.id];
        this.emitEventIfAllPeersWaiting();
    }
    handlePeersChanged() {
        return __awaiter(this, void 0, void 0, function* () {
            this.logDebug('peers changed');
            yield this.clearDeadWaitingPeers();
            yield this.emitEventIfAllPeersWaiting();
        });
    }
    emitEventIfAllPeersWaiting() {
        return __awaiter(this, void 0, void 0, function* () {
            if (yield this.allPeersWaiting()) {
                this.emitter.emit(Event.AllPeersWaiting);
            }
        });
    }
    clearDeadWaitingPeers() {
        return __awaiter(this, void 0, void 0, function* () {
            const me = this.group.getMe();
            const peers = this.group.getPeers();
            const groupPeerIds = peers.map((peer) => peer.id);
            // remove dead peers from the list of waiting peers
            for (const id of Object.keys(this.waitingPeers)) {
                if (id !== me.id && !groupPeerIds.includes(id)) {
                    this.logDebug(`peer ${id} dead, removed from waiting peers`);
                    delete this.waitingPeers[id];
                }
            }
        });
    }
    sendQueryMessage() {
        return __awaiter(this, void 0, void 0, function* () {
            const message = {
                type: MessageType.Query,
            };
            this.logDebug('send query message', JSON.stringify(message));
            yield this.group.sendGroupDataMessage(message);
        });
    }
    sendWaitMessage(data) {
        return __awaiter(this, void 0, void 0, function* () {
            const message = {
                type: MessageType.Wait,
                data,
            };
            this.logDebug('send wait message', JSON.stringify(message));
            yield this.group.sendGroupDataMessage(message);
        });
    }
    sendNotWaitingMessage() {
        return __awaiter(this, void 0, void 0, function* () {
            const message = {
                type: MessageType.NotWaiting,
            };
            yield this.group.sendGroupDataMessage(message);
        });
    }
    waitForAllPeers() {
        return new Promise((resolve, reject) => __awaiter(this, void 0, void 0, function* () {
            let resolveWithMasterDataAndReset;
            let rejectWithCanceledError;
            const cleanupEventListeners = () => {
                this.emitter.removeListener(Event.AllPeersWaiting, resolveWithMasterDataAndReset);
                this.emitter.removeListener(Event.WaitCanceled, rejectWithCanceledError);
            };
            resolveWithMasterDataAndReset = () => __awaiter(this, void 0, void 0, function* () {
                cleanupEventListeners();
                this.logDebug('all peers waiting, resolve');
                const data = yield this.getDataOfWaitingMaster();
                this.reset();
                resolve(data);
            });
            rejectWithCanceledError = () => __awaiter(this, void 0, void 0, function* () {
                cleanupEventListeners();
                this.logDebug('wait will be rejected because it was canceled');
                reject(new Error('Wait canceled'));
            });
            if (yield this.allPeersWaiting()) {
                yield resolveWithMasterDataAndReset();
                return;
            }
            const peers = yield this.group.getPeers();
            const peersLeftToWaitCount = yield this.countPeersLeftToWait();
            this.logDebug(`waiting for ${peersLeftToWaitCount} peers, total ${peers.length} peers in group`);
            this.emitter.once(Event.AllPeersWaiting, resolveWithMasterDataAndReset);
            this.emitter.once(Event.WaitCanceled, rejectWithCanceledError);
        }));
    }
    countPeersLeftToWait() {
        return __awaiter(this, void 0, void 0, function* () {
            const peers = yield this.group.getPeers();
            const peersCount = peers.length + 1; // +1 to count me as well
            const waitingPeersCount = Object.keys(this.waitingPeers).length;
            this.logDebug(`countPeersLeftToWait peers=${peers.length}, total=${peersCount}, waiting=${waitingPeersCount}`);
            return peersCount - waitingPeersCount;
        });
    }
    allPeersWaiting() {
        return __awaiter(this, void 0, void 0, function* () {
            const peersLeftToWaitCount = yield this.countPeersLeftToWait();
            return peersLeftToWaitCount <= 0;
        });
    }
    getDataOfWaitingMaster() {
        return __awaiter(this, void 0, void 0, function* () {
            const master = yield this.group.getMaster();
            return this.waitingPeers[master.id];
        });
    }
    reset() {
        this.waitingPeers = {};
    }
    logDebug(...args) {
        logDebug(this.group.getGroupName(), ...args);
    }
}
exports.WaitService = WaitService;
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], WaitService.prototype, "sendWaitMessageAndWaitForOthers", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], WaitService.prototype, "cancelWait", null);
__decorate([
    (0, lockedDecorator_1.locked)('start', { scope: 'instance' }),
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], WaitService.prototype, "start", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", void 0)
], WaitService.prototype, "stop", null);
//# sourceMappingURL=WaitService.js.map