"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const events_1 = require("events");
const debug_1 = __importDefault(require("debug"));
const DEFAULT_RECONNECT_INTERVAL_MS = 30e3;
const debug = (0, debug_1.default)('@signageos/front-display:Stream:ReconnectStreamPlayer');
class ReconnectStreamPlayer {
    constructor(streamPlayer) {
        this.streamPlayer = streamPlayer;
        this.streamsProperties = {};
    }
    prepare(uri, x, y, width, height, options) {
        debug('prepare', uri, x, y, width, height, options);
        return this.streamPlayer.prepare(uri, x, y, width, height, options);
    }
    play(uri, x, y, width, height, options) {
        return __awaiter(this, void 0, void 0, function* () {
            const streamProperties = this.getStreamProperties(uri, x, y, width, height, options);
            debug('play', streamProperties);
            if ((options === null || options === void 0 ? void 0 : options.autoReconnectInterval) && options.autoReconnectInterval < 10000) {
                throw new Error('Auto reconnect interval must be at least 10000 ms');
            }
            this.streamConnect(streamProperties);
            return streamProperties.emitter;
        });
    }
    stop(uri, x, y, width, height) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            const streamProperties = this.getStreamProperties(uri, x, y, width, height);
            debug('stop', streamProperties);
            (_a = streamProperties.stream) === null || _a === void 0 ? void 0 : _a.removeAllListeners();
            this.cancelStreamReconnectSchedule(streamProperties);
            yield this.streamPlayer.stop(uri, x, y, width, height);
            this.emitEvent('closed', streamProperties);
            this.clearStreamProperties(streamProperties);
        });
    }
    pause(uri, x, y, width, height) {
        return __awaiter(this, void 0, void 0, function* () {
            debug('pause', uri, x, y, width, height);
            yield this.streamPlayer.pause(uri, x, y, width, height);
        });
    }
    resume(uri, x, y, width, height) {
        return __awaiter(this, void 0, void 0, function* () {
            debug('resume', uri, x, y, width, height);
            yield this.streamPlayer.resume(uri, x, y, width, height);
        });
    }
    clearAll() {
        return __awaiter(this, void 0, void 0, function* () {
            Object.keys(this.streamsProperties).forEach((key) => this.cancelStreamReconnectSchedule(this.streamsProperties[key]));
            yield this.streamPlayer.clearAll();
            Object.keys(this.streamsProperties).forEach((key) => this.clearStreamProperties(this.streamsProperties[key]));
        });
    }
    getTracks(videoId) {
        return __awaiter(this, void 0, void 0, function* () {
            return this.streamPlayer.getTracks(videoId);
        });
    }
    selectTrack(videoId, trackType, groupId, trackIndex) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.streamPlayer.selectTrack(videoId, trackType, groupId, trackIndex);
        });
    }
    resetTrack(videoId, trackType, groupId) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.streamPlayer.resetTrack(videoId, trackType, groupId);
        });
    }
    streamConnect(streamProperties) {
        return __awaiter(this, void 0, void 0, function* () {
            debug('stream connect', streamProperties);
            const { uri, x, y, width, height, options } = streamProperties.props;
            try {
                streamProperties.stream = yield this.streamPlayer.play(uri, x, y, width, height, options);
                streamProperties.stream.on('connected', () => {
                    debug('connected', streamProperties);
                    this.emitEvent('connected', streamProperties);
                });
                streamProperties.stream.once('error', () => {
                    debug('error', streamProperties);
                    this.emitEvent('error', streamProperties);
                    this.emitEvent('disconnected', streamProperties);
                    this.scheduleStreamReconnect(streamProperties);
                });
                streamProperties.stream.once('disconnected', () => __awaiter(this, void 0, void 0, function* () {
                    debug('disconnected', streamProperties);
                    this.emitEvent('disconnected', streamProperties);
                    this.scheduleStreamReconnect(streamProperties);
                }));
                streamProperties.stream.on('tracks_changed', (event) => {
                    debug('tracks_changed', streamProperties, event);
                    streamProperties.props.tracks = event.tracks;
                    this.emitEvent('tracks_changed', streamProperties);
                });
            }
            catch (_a) {
                debug('error', streamProperties);
                this.scheduleStreamReconnect(streamProperties);
            }
        });
    }
    scheduleStreamReconnect(streamProperties) {
        var _a, _b, _c, _d;
        const autoReconnectEnabled = (_b = (_a = streamProperties.props.options) === null || _a === void 0 ? void 0 : _a.autoReconnect) !== null && _b !== void 0 ? _b : false;
        if (autoReconnectEnabled) {
            const reconnectIntervalMs = (_d = (_c = streamProperties.props.options) === null || _c === void 0 ? void 0 : _c.autoReconnectInterval) !== null && _d !== void 0 ? _d : DEFAULT_RECONNECT_INTERVAL_MS;
            if (!streamProperties.reconnectTimeoutHandler) {
                streamProperties.reconnectTimeoutHandler = setTimeout(() => {
                    delete streamProperties.reconnectTimeoutHandler;
                    this.streamReconnect(streamProperties);
                }, reconnectIntervalMs);
            }
            else {
                console.warn('Reconnect is already scheduled', streamProperties.props);
            }
        }
    }
    streamReconnect(streamProperties) {
        return __awaiter(this, void 0, void 0, function* () {
            const { uri, x, y, width, height } = streamProperties.props;
            this.emitEvent('reconnect', streamProperties);
            yield this.streamPlayer.stop(uri, x, y, width, height);
            yield this.streamConnect(streamProperties);
        });
    }
    getKeyForProperties(uri, x, y, width, height) {
        return `${uri}_${x}_${y}_${width}_${height}`;
    }
    getStreamProperties(uri, x, y, width, height, options) {
        const propertiesKey = this.getKeyForProperties(uri, x, y, width, height);
        if (!this.streamsProperties.hasOwnProperty(propertiesKey)) {
            this.streamsProperties[propertiesKey] = {
                props: { uri, x, y, width, height, options },
                emitter: new events_1.EventEmitter(),
            };
            this.streamsProperties[propertiesKey].emitter.on('error', () => {
                /* making it safe */
            });
        }
        return this.streamsProperties[propertiesKey];
    }
    cancelStreamReconnectSchedule(streamProperties) {
        var _a;
        const { uri, x, y, width, height } = streamProperties.props;
        const propertiesKey = this.getKeyForProperties(uri, x, y, width, height);
        if ((_a = this.streamsProperties[propertiesKey]) === null || _a === void 0 ? void 0 : _a.reconnectTimeoutHandler) {
            clearTimeout(this.streamsProperties[propertiesKey].reconnectTimeoutHandler);
            delete this.streamsProperties[propertiesKey].reconnectTimeoutHandler;
        }
    }
    clearStreamProperties(streamProperties) {
        const { uri, x, y, width, height } = streamProperties.props;
        const propertiesKey = this.getKeyForProperties(uri, x, y, width, height);
        if (this.streamsProperties.hasOwnProperty(propertiesKey)) {
            this.cancelStreamReconnectSchedule(streamProperties);
            this.streamsProperties[propertiesKey].emitter.removeAllListeners();
            delete this.streamsProperties[propertiesKey];
        }
    }
    emitEvent(type, streamProperties) {
        const { uri, x, y, width, height, options, tracks } = streamProperties.props;
        const { emitter } = streamProperties;
        switch (type) {
            case 'error':
                emitter.emit('error', {
                    type: 'error',
                    uri,
                    x,
                    y,
                    width,
                    height,
                    protocol: options === null || options === void 0 ? void 0 : options.protocol,
                    errorMessage: 'Stream playback failed',
                });
                break;
            case 'connected':
                emitter.emit('connected', {
                    type: 'connected',
                    uri,
                    x,
                    y,
                    width,
                    height,
                });
                break;
            case 'disconnected':
                emitter.emit('disconnected', {
                    type: 'disconnected',
                    uri,
                    x,
                    y,
                    width,
                    height,
                });
                break;
            case 'closed':
                emitter.emit('closed', {
                    type: 'closed',
                    uri,
                    x,
                    y,
                    width,
                    height,
                });
                break;
            case 'reconnect':
                emitter.emit('reconnect', {
                    type: 'reconnect',
                    uri,
                    x,
                    y,
                    width,
                    height,
                });
                break;
            case 'tracks_changed':
                emitter.emit('tracks_changed', {
                    type: 'tracks_changed',
                    uri,
                    x,
                    y,
                    width,
                    height,
                    tracks,
                });
                break;
            default:
                break;
        }
    }
}
exports.default = ReconnectStreamPlayer;
//# sourceMappingURL=ReconnectStreamPlayer.js.map