import { IVideoSrcArguments } from '../Video/IVideoSrcArguments';
import IStream from './IStream';
import StreamProtocol from './StreamProtocol';
export type TrackType = 'AUDIO' | 'TEXT' | 'VIDEO';
export type DrmScheme = 'CommonPSSH' | 'ClearKey' | 'Widevine' | 'PlayReady' | string;
export type ITrackInfo = ITrackVideoInfo | ITrackAudioInfo | ITrackTextInfo;
export interface ITrack<T extends TrackType> {
    trackType: T;
    mimeType: string;
    groupId: string;
    trackIndex: number;
    selected: boolean;
    language: string | null;
    supported: boolean;
}
export interface ITrackVideoInfo extends ITrack<'VIDEO'> {
    videoSize: {
        width: number;
        height: number;
    };
}
export interface ITrackAudioInfo extends ITrack<'AUDIO'> {
    channelCount: number;
}
export interface ITrackTextInfo extends ITrack<'TEXT'> {
    selection: 'default' | 'forced' | 'autoselect';
}
export interface IStreamOptions {
    protocol?: keyof typeof StreamProtocol | string;
    /**
     * If true, the stream will be prepared in the background.
     */
    background?: boolean;
    /**
     * If true, the stream will automatically reconnect when the connection is lost.
     */
    autoReconnect?: boolean;
    /**
     * Interval in milliseconds to try to reconnect to the stream.
     */
    autoReconnectInterval?: number;
}
export interface IStreamPrepareOptions extends IStreamOptions {
    trackSelection?: {
        maxAudioChannelCount?: number;
        minVideoSize?: {
            width: number;
            height: number;
        };
        maxVideoSize?: {
            width: number;
            height: number;
        };
        preferredAudioLanguages?: string[];
        preferredTextLanguages?: string[];
    };
    drm?: {
        scheme: string;
        licenseUri: string;
        licenseRequestHeaders: {
            [key: string]: string;
        };
    };
}
interface IStreamPlayer {
    prepare(uri: string, x: number, y: number, width: number, height: number, options?: IStreamPrepareOptions): Promise<void>;
    play(uri: string, x: number, y: number, width: number, height: number, options?: IStreamOptions): Promise<IStream>;
    stop(uri: string, x: number, y: number, width: number, height: number): Promise<void>;
    pause(uri: string, x: number, y: number, width: number, height: number): Promise<void>;
    resume(uri: string, x: number, y: number, width: number, height: number): Promise<void>;
    clearAll(): Promise<void>;
    getTracks(videoId: IVideoSrcArguments): Promise<ITrackInfo[]>;
    selectTrack(videoId: IVideoSrcArguments, trackType: TrackType, groupId: string, trackIndex: number): Promise<void>;
    resetTrack(videoId: IVideoSrcArguments, trackType: TrackType, groupId?: string): Promise<void>;
}
export default IStreamPlayer;
