"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PeerDiscoveryService = void 0;
const events_1 = require("events");
const debug_1 = __importDefault(require("debug"));
const debugDecorator_1 = require("@signageos/lib/dist/Debug/debugDecorator");
const IPeerDiscoveryService_1 = require("./IPeerDiscoveryService");
const PeerDiscoveryServiceMessageHandler_1 = require("./PeerDiscoveryServiceMessageHandler");
const DEBUG_NAMESPACE = '@signageos/front-display:PeerNetwork:PeerDiscoveryService';
const logDebug = (0, debug_1.default)(DEBUG_NAMESPACE);
const defaultConfig = {
    pingIntervalMs: 5e3,
    cleanDeadPeersIntervalMs: 30e3,
    aliveTimeout: 30e3,
};
/**
 * Handles discovery of other peers in the network and communication with them.
 */
class PeerDiscoveryService {
    constructor(id, socketFactory, getNetworkInfo, config = {}) {
        this.id = id;
        this.socketFactory = socketFactory;
        this.getNetworkInfo = getNetworkInfo;
        this.peers = {};
        this.emitter = new events_1.EventEmitter();
        this.pingInterval = null;
        this.cleanInterval = null;
        logDebug('initiated with id ' + this.id);
        this.config = Object.assign(Object.assign({}, defaultConfig), config);
        this.messageHandler = new PeerDiscoveryServiceMessageHandler_1.PeerDiscoveryServiceMessageSender(this.id, this.socketFactory);
        this.registerMessageHandlerListeners();
    }
    start() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.messageHandler.isStarted()) {
                throw new Error('PeerDiscoveryService already started');
            }
            const networkInfo = yield this.getNetworkInfo();
            if (!networkInfo) {
                throw new Error('Cannot obtain network info');
            }
            yield this.messageHandler.start();
            try {
                yield this.messageHandler.sendQuery();
                yield this.messageHandler.sendAnnounce(networkInfo.address, networkInfo.port);
            }
            catch (error) {
                console.error('PeerDiscoveryService: Failed to send initial messages query and announce', error);
            }
            this.pingInterval = setInterval(() => this.messageHandler.sendPing(), this.config.pingIntervalMs);
            this.cleanInterval = setInterval(() => this.cleanDeadPeers(), this.config.cleanDeadPeersIntervalMs);
        });
    }
    stop() {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.messageHandler.isStarted()) {
                throw new Error('PeerDiscoveryService already stopped');
            }
            yield this.messageHandler.sendRenounce();
            if (this.pingInterval) {
                clearInterval(this.pingInterval);
                this.pingInterval = null;
            }
            if (this.cleanInterval) {
                clearInterval(this.cleanInterval);
                this.cleanInterval = null;
            }
            yield this.messageHandler.stop();
        });
    }
    isStarted() {
        return this.messageHandler.isStarted();
    }
    getMe() {
        return {
            id: this.id,
            aliveAt: new Date(),
        };
    }
    getPeers() {
        return Object.keys(this.peers).map((key) => this.peers[key]);
    }
    addListener(event, callback) {
        this.emitter.addListener(event, callback);
    }
    removeListener(event, callback) {
        this.emitter.removeListener(event, callback);
    }
    registerMessageHandlerListeners() {
        this.messageHandler.onQuery((message) => this.handleQueryMessage(message.sourceId));
        this.messageHandler.onAnnounce((message) => this.handleAnnounceMessage({
            sourceId: message.sourceId,
            address: message.address,
            port: message.port,
        }));
        this.messageHandler.onRenounce((message) => this.handleRenounce(message.sourceId));
        this.messageHandler.onPing((message) => this.handlePingMessage(message.sourceId));
    }
    handleQueryMessage(sourceId) {
        return __awaiter(this, void 0, void 0, function* () {
            logDebug('received query message from ' + sourceId);
            const networkInfo = yield this.getNetworkInfo();
            if (!networkInfo) {
                logDebug('Cannot obtain network info. Query message will be discarded without reply.');
                return;
            }
            yield this.messageHandler.sendAnnounce(networkInfo.address, networkInfo.port);
        });
    }
    handleAnnounceMessage({ sourceId, address, port }) {
        logDebug('received announce message from ' + sourceId, { address, port });
        const peer = {
            id: sourceId,
            aliveAt: new Date(),
            address,
            port,
        };
        const oldPeer = this.peers[sourceId];
        if (!oldPeer || oldPeer.address !== address || oldPeer.port !== port) {
            if (oldPeer) {
                logDebug(`Peer ${oldPeer.id} changed`, peer);
            }
            else {
                logDebug(`New peer ${peer.id}`, peer);
            }
            this.peers[sourceId] = peer;
            logDebug('Peers changed', this.peers);
            this.emitter.emit(IPeerDiscoveryService_1.PeerDiscoveryServiceEvent.PeersChanged);
        }
    }
    handleRenounce(sourceId) {
        logDebug('received renounce message from ' + sourceId);
        if (this.peers[sourceId] !== undefined) {
            logDebug(`Peer ${sourceId} renounced`);
            delete this.peers[sourceId];
            logDebug('Peers changed', this.peers);
            this.emitter.emit(IPeerDiscoveryService_1.PeerDiscoveryServiceEvent.PeersChanged);
        }
    }
    handlePingMessage(sourceId) {
        return __awaiter(this, void 0, void 0, function* () {
            logDebug('received ping message from ' + sourceId);
            if (this.peers[sourceId] !== undefined) {
                this.peers[sourceId].aliveAt = new Date();
            }
            else {
                yield this.messageHandler.sendQuery();
            }
        });
    }
    cleanDeadPeers() {
        for (const peerId in this.peers) {
            const aliveAt = this.peers[peerId].aliveAt;
            const deadAt = aliveAt.valueOf() + this.config.aliveTimeout;
            const now = Date.now();
            if (deadAt <= now) {
                logDebug(`${peerId} is dead, renouncing`);
                this.handleRenounce(peerId);
            }
        }
    }
}
exports.PeerDiscoveryService = PeerDiscoveryService;
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], PeerDiscoveryService.prototype, "start", null);
__decorate([
    (0, debugDecorator_1.debug)(DEBUG_NAMESPACE),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], PeerDiscoveryService.prototype, "stop", null);
//# sourceMappingURL=PeerDiscoveryService.js.map