"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const async_lock_1 = __importDefault(require("async-lock"));
const path_1 = require("path");
const lodash_1 = __importDefault(require("lodash"));
const timer_1 = require("@signageos/lib/dist/DateTime/timer");
const FileNotFoundError_1 = __importDefault(require("../Front/Applet/Error/FileNotFoundError"));
const ErrorCodes_1 = __importDefault(require("../Front/Applet/Error/ErrorCodes"));
const ErrorSuggestions_1 = __importDefault(require("../Front/Applet/Error/ErrorSuggestions"));
const DATA_PATH_PREFIX = 'data';
class OfflineCache {
    constructor(fileSystem) {
        this.fileSystem = fileSystem;
        this.storageUnitsCache = null;
        this.asyncLock = new async_lock_1.default();
        this.fileSystem.onStorageUnitsChanged(() => (this.storageUnitsCache = null));
        setInterval(() => (this.storageUnitsCache = null), 30e3);
    }
    retriableDownloadFile(countOfRetrials, uid, uri, headers, forceInternalStorageUnit = false) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const storageUnit = yield this.getMostFreeStorageUnit(forceInternalStorageUnit);
                const targetFilePath = { storageUnit, filePath: this.addDataPrefixToUid(uid) };
                yield this.ensureDirectory({ storageUnit, filePath: this.getParentDirectoryPath(targetFilePath.filePath) });
                yield this.fileSystem.downloadFile(targetFilePath, uri, headers);
            }
            catch (error) {
                console.info('Failed to download ' + uid + '. Retrying ' + countOfRetrials, error);
                if (countOfRetrials > 0) {
                    const defferSeconds = Math.pow(2, Math.max(5 - countOfRetrials, 0));
                    yield (0, timer_1.wait)(window, defferSeconds * 1e3);
                    yield this.retriableDownloadFile(countOfRetrials - 1, uid, uri, headers);
                }
                else {
                    throw error;
                }
            }
        });
    }
    fileExists(uid) {
        return __awaiter(this, void 0, void 0, function* () {
            const prefixedPath = this.addDataPrefixToUid(uid);
            const storageUnits = yield this.getStorageUnits();
            for (const storageUnit of storageUnits) {
                if (yield this.fileSystem.exists({ storageUnit, filePath: prefixedPath })) {
                    return true;
                }
            }
            return false;
        });
    }
    fileIsDirectory(uid) {
        return __awaiter(this, void 0, void 0, function* () {
            const prefixedPath = this.addDataPrefixToUid(uid);
            const storageUnits = yield this.getStorageUnits();
            for (const storageUnit of storageUnits) {
                if (yield this.fileSystem.exists({ storageUnit, filePath: prefixedPath })) {
                    return yield this.fileSystem.isDirectory({ storageUnit, filePath: prefixedPath });
                }
            }
            throw this.createFileNotFoundError(uid);
        });
    }
    getFile(uid) {
        return __awaiter(this, void 0, void 0, function* () {
            const prefixedPath = this.addDataPrefixToUid(uid);
            const storageUnits = yield this.getStorageUnits();
            for (const storageUnit of storageUnits) {
                const filePath = { storageUnit, filePath: prefixedPath };
                if (yield this.fileSystem.exists(filePath)) {
                    return yield this.fileSystem.getFile(filePath);
                }
            }
            throw this.createFileNotFoundError(uid);
        });
    }
    getFullFilePath(uid) {
        return __awaiter(this, void 0, void 0, function* () {
            const prefixedPath = this.addDataPrefixToUid(uid);
            const storageUnits = yield this.getStorageUnits();
            for (const storageUnit of storageUnits) {
                const filePath = { storageUnit, filePath: prefixedPath };
                if (yield this.fileSystem.exists(filePath)) {
                    return filePath;
                }
            }
            throw this.createFileNotFoundError(uid);
        });
    }
    readFile(uid) {
        return __awaiter(this, void 0, void 0, function* () {
            const prefixedPath = this.addDataPrefixToUid(uid);
            const storageUnits = yield this.getStorageUnits();
            for (const storageUnit of storageUnits) {
                const filePath = { storageUnit, filePath: prefixedPath };
                if (yield this.fileSystem.exists(filePath)) {
                    return yield this.fileSystem.readFile(filePath);
                }
            }
            throw this.createFileNotFoundError(uid);
        });
    }
    deleteFile(uid, recursive) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!(yield this.fileExists(uid))) {
                throw this.createFileNotFoundError(uid);
            }
            const prefixedPath = this.addDataPrefixToUid(uid);
            const storageUnits = yield this.getStorageUnits();
            for (const storageUnit of storageUnits) {
                const filePath = { storageUnit, filePath: prefixedPath };
                if (yield this.fileSystem.exists(filePath)) {
                    yield this.fileSystem.deleteFile(filePath, recursive);
                }
            }
        });
    }
    deleteFileAndDeleteDirectoryIfEmpty(uid, recursive = false) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.deleteFile(uid, recursive);
            const parentDirectoryUid = this.getParentDirectoryPath(uid);
            if (parentDirectoryUid !== '') {
                // do not delete root directory
                yield this.deleteDirectoryIfEmpty(parentDirectoryUid);
            }
        });
    }
    getFileChecksum(uid, hashType) {
        return __awaiter(this, void 0, void 0, function* () {
            const prefixedPath = this.addDataPrefixToUid(uid);
            const storageUnits = yield this.getStorageUnits();
            for (const storageUnit of storageUnits) {
                const filePath = { storageUnit, filePath: prefixedPath };
                if (yield this.fileSystem.exists(filePath)) {
                    return yield this.fileSystem.getFileChecksum(filePath, hashType);
                }
            }
            throw this.createFileNotFoundError(uid);
        });
    }
    extractFile(archiveUid, destinationDirectoryUid, method, forceInternalStorageUnit = false) {
        return __awaiter(this, void 0, void 0, function* () {
            const prefixedArchivePath = this.addDataPrefixToUid(archiveUid);
            const prefixedDestinationDirectoryPath = this.addDataPrefixToUid(destinationDirectoryUid);
            const storageUnits = yield this.getStorageUnits();
            for (const storageUnit of storageUnits) {
                const archiveFilePath = { storageUnit, filePath: prefixedArchivePath };
                if (yield this.fileSystem.exists(archiveFilePath)) {
                    const destinationStorageUnit = yield this.getMostFreeStorageUnit(forceInternalStorageUnit);
                    const destinationDirectoryPath = { storageUnit: destinationStorageUnit, filePath: prefixedDestinationDirectoryPath };
                    const destinationFileUids = yield this.listFilesRecursively(destinationDirectoryUid);
                    if (yield this.fileExists(destinationDirectoryUid)) {
                        yield Promise.all(destinationFileUids.map((destinationFileUid) => this.deleteFileAndDeleteDirectoryIfEmpty(destinationFileUid)));
                    }
                    yield this.ensureDirectory(destinationDirectoryPath);
                    return yield this.fileSystem.extractFile(archiveFilePath, destinationDirectoryPath, method);
                }
            }
            throw new FileNotFoundError_1.default({
                kind: 'fileNotFoundError',
                message: `Archive file ${archiveUid} was not found`,
                code: ErrorCodes_1.default.FILE_NOT_FOUND,
                suggestion: ErrorSuggestions_1.default.FILE_NOT_FOUND,
            });
        });
    }
    getArchiveInfo(uid) {
        return __awaiter(this, void 0, void 0, function* () {
            const prefixedPath = this.addDataPrefixToUid(uid);
            const storageUnits = yield this.getStorageUnits();
            for (const storageUnit of storageUnits) {
                const filePath = { storageUnit, filePath: prefixedPath };
                if (yield this.fileSystem.exists(filePath)) {
                    return yield this.fileSystem.getArchiveInfo(filePath);
                }
            }
            throw this.createFileNotFoundError(uid);
        });
    }
    listFilesRecursively(directoryUid = '') {
        return __awaiter(this, void 0, void 0, function* () {
            if (!(yield this.fileExists(directoryUid))) {
                return [];
            }
            const fileUids = yield this.listFileUids(directoryUid);
            const recursiveFileUids = fileUids.map((fileUid) => __awaiter(this, void 0, void 0, function* () {
                if (yield this.fileIsDirectory(fileUid)) {
                    return yield this.listFilesRecursively(fileUid);
                }
                else {
                    return fileUid;
                }
            }));
            return lodash_1.default.flatten(yield Promise.all(recursiveFileUids));
        });
    }
    listFileUids(directoryUid) {
        return __awaiter(this, void 0, void 0, function* () {
            const prefixedDirectoryPath = this.addDataPrefixToUid(directoryUid);
            const storageUnits = yield this.getStorageUnits();
            return lodash_1.default.uniq(lodash_1.default.flatten(yield Promise.all(storageUnits.map((storageUnit) => __awaiter(this, void 0, void 0, function* () {
                const directoryPath = { storageUnit, filePath: prefixedDirectoryPath };
                if (yield this.fileSystem.exists(directoryPath)) {
                    const filePaths = yield this.fileSystem.listFiles(directoryPath);
                    return filePaths.map((filePath) => this.stripDataPrefixFromPath(filePath.filePath));
                }
                else {
                    return [];
                }
            })))));
        });
    }
    deleteDirectoryIfEmpty(uid) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.asyncLock.acquire(`cleanupDirectory.${uid}`, () => __awaiter(this, void 0, void 0, function* () {
                if (yield this.fileExists(uid)) {
                    const files = yield this.listFileUids(uid);
                    if (files.length === 0) {
                        yield this.deleteFileAndDeleteDirectoryIfEmpty(uid);
                    }
                }
            }));
        });
    }
    ensureDirectory(directoryPath) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!(yield this.fileSystem.exists(directoryPath))) {
                if (!this.isRootFilePath(directoryPath)) {
                    yield this.ensureDirectory({
                        storageUnit: directoryPath.storageUnit,
                        filePath: this.getParentDirectoryPath(directoryPath.filePath),
                    });
                }
                yield this.fileSystem.createDirectory(directoryPath);
            }
        });
    }
    getParentDirectoryPath(directoryPath) {
        const parentDirectoryPath = path_1.posix.dirname(directoryPath);
        if (parentDirectoryPath === '.') {
            return '';
        }
        else {
            return parentDirectoryPath;
        }
    }
    isRootFilePath(filePath) {
        return filePath.filePath === DATA_PATH_PREFIX;
    }
    getMostFreeStorageUnit(onlyInternal = false) {
        return __awaiter(this, void 0, void 0, function* () {
            let storageUnits = yield this.getStorageUnits();
            if (onlyInternal) {
                storageUnits = storageUnits.filter((storageUnit) => !storageUnit.removable);
            }
            return storageUnits.sort((a, b) => b.usableSpace - a.usableSpace)[0];
        });
    }
    getStorageUnits() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.storageUnitsCache !== null) {
                return this.storageUnitsCache;
            }
            const storageUnits = yield this.fileSystem.listStorageUnits();
            this.storageUnitsCache = storageUnits;
            return storageUnits;
        });
    }
    addDataPrefixToUid(filePath) {
        if (filePath === '') {
            return DATA_PATH_PREFIX;
        }
        return DATA_PATH_PREFIX + '/' + filePath;
    }
    stripDataPrefixFromPath(filePath) {
        let uid = filePath;
        if (uid.startsWith(DATA_PATH_PREFIX)) {
            uid = uid.slice(DATA_PATH_PREFIX.length);
        }
        if (uid.startsWith('/')) {
            return uid.slice(1);
        }
        return uid;
    }
    createFileNotFoundError(fileUid) {
        return new FileNotFoundError_1.default({
            kind: 'fileNotFoundError',
            message: `File ${fileUid} was not found`,
            code: ErrorCodes_1.default.FILE_NOT_FOUND,
            suggestion: ErrorSuggestions_1.default.FILE_NOT_FOUND,
        });
    }
}
exports.default = OfflineCache;
//# sourceMappingURL=OfflineCache.js.map