"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lodash_1 = require("lodash");
/**
 * An ISO 8061 time-zone offset is the amount of time that a time-zone differs from Greenwich/UTC.
 * This is usually a fixed number of hours and minutes.
 *
 * Negative UTC offsets describe a time zone west of UTC±00:00, where the civil time is behind (or earlier) than UTC
 * so the zone designator will look like "−03:00","−0300", or "−03".
 *
 * Positive UTC offsets describe a time zone at or east of UTC±00:00, where the civil time is the same as or
 * ahead (or later) than UTC so the zone designator will look like "+02:00","+0200", or "+02".
 *
 * In 2008, time-zone offsets around the world extended from -12:00 to +14:00.
 * To prevent any problems with that range being extended, yet still provide validation, the range of offsets
 * is restricted to -18:00 to 18:00 inclusive.
 */
class ZoneOffset {
    static fromTime(hours, minutes = 0) {
        if (minutes < 0 || minutes > 59) {
            throw new Error(`Minutes must be between 0 and 59 inclusive`);
        }
        if (hours < 0) {
            return new ZoneOffset(hours * 60 - minutes);
        }
        return new ZoneOffset(hours * 60 + minutes);
    }
    /**
     * A negative offset indicates a time zone east of the Prime Meridian. A positive offset indicates the time zone
     * west of the Prime Meridian.
     */
    static fromPosix(reverseMinutes) {
        return new ZoneOffset(-reverseMinutes);
    }
    constructor(totalMinutes) {
        this.totalMinutes = totalMinutes;
        if (totalMinutes < ZoneOffset.MIN_RANGE_TOTAL_MINUTES || totalMinutes > ZoneOffset.MAX_RANGE_TOTAL_MINUTES) {
            throw new Error(`Offset in minutes must be within ±${ZoneOffset.MAX_RANGE_TOTAL_MINUTES}.`);
        }
    }
    /**
     * A negative offset indicates a time zone east of the Prime Meridian. A positive offset indicates the time zone
     * west of the Prime Meridian.
     */
    toPosix() {
        return -this.totalMinutes;
    }
    toString() {
        const hours = Math.floor(this.totalMinutes / 60).toString(10);
        const minutes = (this.totalMinutes % 60).toString(10);
        return `${this.totalMinutes < 0 ? '-' : '+'}${(0, lodash_1.padStart)(hours, 2, '0')}:${(0, lodash_1.padStart)(minutes, 2, '0')}`;
    }
}
ZoneOffset.MIN_RANGE_TOTAL_MINUTES = -1080;
ZoneOffset.MAX_RANGE_TOTAL_MINUTES = 1080;
exports.default = ZoneOffset;
//# sourceMappingURL=ZoneOffset.js.map