"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const events_1 = require("events");
const cssRotationHelper_1 = require("./cssRotationHelper");
/**
 * Wrapper class for any implementation of video player that converts CSS-rotated coordinates back to landscape for the system
 *
 * When the content is rotated via CSS, all the coordinates that are sent from it are rotated as well.
 * That's bad because from the point of view of the system it's landscape and the rotated coordinates don't make sense.
 * This is a proxy class that converts the coordinates between the client and the system so neither of them have to worry about it.
 */
class CSSRotationVideoPlayerAdapter {
    constructor(window, videoPlayer, getAngle) {
        this.window = window;
        this.videoPlayer = videoPlayer;
        this.getAngle = getAngle;
    }
    getMaxVideoCount() {
        return this.videoPlayer.getMaxVideoCount();
    }
    prepare(uri, x, y, width, height, options) {
        return __awaiter(this, void 0, void 0, function* () {
            const angle = yield this.getAngle();
            const coords = (0, cssRotationHelper_1.convertClientCoordinatesToSystem)(this.window, angle, x, y, width, height);
            yield this.videoPlayer.prepare(uri, coords.x, coords.y, coords.width, coords.height, options);
        });
    }
    play(uri, x, y, width, height) {
        return __awaiter(this, void 0, void 0, function* () {
            const angle = yield this.getAngle();
            const coords = (0, cssRotationHelper_1.convertClientCoordinatesToSystem)(this.window, angle, x, y, width, height);
            const video = yield this.videoPlayer.play(uri, coords.x, coords.y, coords.width, coords.height);
            return this.convertEventEmitterWithConvertedCoordinatesBackToOriginalCoordinates(video, x, y, width, height);
        });
    }
    stop(uri, x, y, width, height) {
        return __awaiter(this, void 0, void 0, function* () {
            const angle = yield this.getAngle();
            const coords = (0, cssRotationHelper_1.convertClientCoordinatesToSystem)(this.window, angle, x, y, width, height);
            yield this.videoPlayer.stop(uri, coords.x, coords.y, coords.width, coords.height);
        });
    }
    pause(uri, x, y, width, height) {
        return __awaiter(this, void 0, void 0, function* () {
            const angle = yield this.getAngle();
            const coords = (0, cssRotationHelper_1.convertClientCoordinatesToSystem)(this.window, angle, x, y, width, height);
            yield this.videoPlayer.pause(uri, coords.x, coords.y, coords.width, coords.height);
        });
    }
    resume(uri, x, y, width, height) {
        return __awaiter(this, void 0, void 0, function* () {
            const angle = yield this.getAngle();
            const coords = (0, cssRotationHelper_1.convertClientCoordinatesToSystem)(this.window, angle, x, y, width, height);
            yield this.videoPlayer.resume(uri, coords.x, coords.y, coords.width, coords.height);
        });
    }
    clearAll() {
        return this.videoPlayer.clearAll();
    }
    getDuration(uri, x, y, width, height) {
        return __awaiter(this, void 0, void 0, function* () {
            const angle = yield this.getAngle();
            const coords = (0, cssRotationHelper_1.convertClientCoordinatesToSystem)(this.window, angle, x, y, width, height);
            return this.videoPlayer.getDuration(uri, coords.x, coords.y, coords.width, coords.height);
        });
    }
    convertEventEmitterWithConvertedCoordinatesBackToOriginalCoordinates(videoEmitter, originalX, originalY, originalWidth, originalHeight) {
        const convertedVideoEmitter = new events_1.EventEmitter();
        const convertEvent = (event) => (Object.assign(Object.assign({}, event), { srcArguments: {
                uri: event.srcArguments.uri,
                x: originalX,
                y: originalY,
                width: originalWidth,
                height: originalHeight,
            } }));
        videoEmitter.on('playing', (event) => convertedVideoEmitter.emit('playing', convertEvent(event)));
        videoEmitter.on('ended', (event) => convertedVideoEmitter.emit('ended', convertEvent(event)));
        videoEmitter.on('error', (event) => convertedVideoEmitter.emit('error', convertEvent(event)));
        videoEmitter.on('stopped', (event) => convertedVideoEmitter.emit('stopped', convertEvent(event)));
        // "error" event type is treated as a special case and has to have at least one listener or it can crash the whole process
        // https://nodejs.org/api/events.html#events_error_events
        convertedVideoEmitter.on('error', () => {
            /* do nothing */
        });
        return convertedVideoEmitter;
    }
}
exports.default = CSSRotationVideoPlayerAdapter;
//# sourceMappingURL=CSSRotationVideoPlayerAdapter.js.map