"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const jose = __importStar(require("jose"));
const buffer_1 = require("buffer");
const KEY_ALGORITHM = 'RSA-OAEP';
const KEY_SIZE = 4096;
const KEY_HASH = 'SHA-256';
const KEY_ALIAS = 'SecretManager';
const KEY_VALIDITY_MONTHS = 6;
const JWE_ENC = 'A256GCM';
class SubtleCryptoSecretManager {
    constructor(subtleCrypto, storage) {
        this.subtleCrypto = subtleCrypto;
        this.storage = storage;
    }
    generateKeys() {
        return __awaiter(this, void 0, void 0, function* () {
            const keyPair = yield this.subtleCrypto().generateKey({
                name: KEY_ALGORITHM,
                modulusLength: KEY_SIZE,
                publicExponent: new Uint8Array([1, 0, 1]),
                hash: KEY_HASH,
            }, true, ['encrypt', 'decrypt']);
            yield this.saveKeyPair(keyPair);
        });
    }
    encryptUtf8ToJweCompact(text) {
        return __awaiter(this, void 0, void 0, function* () {
            const bytes = buffer_1.Buffer.from(text, 'utf8');
            return this.encryptBytesToJweCompact(bytes);
        });
    }
    decryptJweCompactToUtf8(jweCompact) {
        return __awaiter(this, void 0, void 0, function* () {
            const plaintext = yield this.decryptJweCompactToBytes(jweCompact);
            return buffer_1.Buffer.from(plaintext).toString('utf-8');
        });
    }
    decryptJweCompactToBytes(jweCompact) {
        return __awaiter(this, void 0, void 0, function* () {
            const privateKey = yield this.requirePrivateKey();
            const { plaintext } = yield jose.compactDecrypt(jweCompact, privateKey);
            return plaintext;
        });
    }
    encryptBytesToJweCompact(value) {
        return __awaiter(this, void 0, void 0, function* () {
            const [publicKey, alg] = yield this.requirePublicKey();
            return yield new jose.CompactEncrypt(value).setProtectedHeader({ alg, enc: JWE_ENC }).encrypt(publicKey);
        });
    }
    encryptUtf8ToJweGeneral(text) {
        return __awaiter(this, void 0, void 0, function* () {
            const bytes = buffer_1.Buffer.from(text, 'utf8');
            return this.encryptBytesToJweGeneral(bytes);
        });
    }
    decryptJweGeneralToUtf8(jweGeneral) {
        return __awaiter(this, void 0, void 0, function* () {
            const plaintext = yield this.decryptJweGeneralToBytes(jweGeneral);
            return buffer_1.Buffer.from(plaintext).toString('utf-8');
        });
    }
    decryptJweGeneralToBytes(jweGeneral) {
        return __awaiter(this, void 0, void 0, function* () {
            const privateKey = yield this.requirePrivateKey();
            const { plaintext } = yield jose.generalDecrypt(jweGeneral, privateKey);
            return plaintext;
        });
    }
    encryptBytesToJweGeneral(value) {
        return __awaiter(this, void 0, void 0, function* () {
            const [publicKey, alg] = yield this.requirePublicKey();
            return yield new jose.GeneralEncrypt(value)
                .setProtectedHeader({ enc: JWE_ENC })
                .addRecipient(publicKey)
                .setUnprotectedHeader({ alg })
                .encrypt();
        });
    }
    getPublicKey() {
        return __awaiter(this, void 0, void 0, function* () {
            const keyPair = yield this.requireKeyPair();
            return {
                algorithm: keyPair.algorithm,
                publicKeySpki: keyPair.publicKeySpki,
                keyValidity: keyPair.keyValidity,
            };
        });
    }
    saveKeyPair(keyPair) {
        return __awaiter(this, void 0, void 0, function* () {
            const algorithm = keyPair.publicKey.algorithm;
            if (!algorithm.name.startsWith('RSA-')) {
                throw new Error(`Unsupported algorithm: ${algorithm.name}`);
            }
            const publicKeySpki = yield jose.exportSPKI(keyPair.publicKey);
            const privateKeyPkcs8 = yield jose.exportPKCS8(keyPair.privateKey);
            const actualDate = new Date();
            const validUntil = new Date(actualDate);
            validUntil.setMonth(actualDate.getMonth() + KEY_VALIDITY_MONTHS);
            const keyData = {
                algorithm,
                publicKeySpki,
                privateKeyPkcs8,
                keyValidity: {
                    start: actualDate,
                    end: validUntil,
                },
            };
            yield this.storage.saveKeyPair(KEY_ALIAS, keyData);
        });
    }
    requirePrivateKey() {
        return __awaiter(this, void 0, void 0, function* () {
            const keyPair = yield this.requireKeyPair();
            const alg = buildJweAlg(keyPair.algorithm);
            return yield jose.importPKCS8(keyPair.privateKeyPkcs8, alg);
        });
    }
    requirePublicKey() {
        return __awaiter(this, void 0, void 0, function* () {
            const keyPair = yield this.requireKeyPair();
            const alg = buildJweAlg(keyPair.algorithm);
            return [yield jose.importSPKI(keyPair.publicKeySpki, alg), alg];
        });
    }
    requireKeyPair() {
        return __awaiter(this, void 0, void 0, function* () {
            const keyPair = yield this.storage.getKeyPair(KEY_ALIAS);
            if (!keyPair) {
                throw new Error(`Key not found.`);
            }
            return keyPair;
        });
    }
}
exports.default = SubtleCryptoSecretManager;
const buildJweAlg = (algorithm) => {
    switch (algorithm.name) {
        case 'RSA-OAEP':
            const hash = algorithm.hash.name;
            switch (hash) {
                case 'SHA-1':
                    return algorithm.name;
                default:
                    return `${algorithm.name}-${hash.slice(-3)}`;
            }
        default:
            throw new Error(`Unsupported algorithm: ${algorithm.name}`);
    }
};
//# sourceMappingURL=SubtleCryptoSecretManager.js.map