"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
class DefaultBrowserSecretStorage {
    constructor(storage, getDeviceUid) {
        this.storage = storage;
        this.getDeviceUid = getDeviceUid;
    }
    saveKeyPair(name, keyPair) {
        return __awaiter(this, void 0, void 0, function* () {
            const password = yield this.getDeviceUid();
            const salt = crypto.getRandomValues(new Uint8Array(DefaultBrowserSecretStorage.SALT_LENGTH));
            const key = yield this.deriveKey(password, salt);
            const { encrypted, iv } = yield this.encryptData(key, JSON.stringify(keyPair));
            this.storage.setItem(name, JSON.stringify({
                encrypted: this.uint8ToBase64(this.base64ToUint8(encrypted)),
                iv: this.uint8ToBase64(this.base64ToUint8(iv)),
                salt: this.uint8ToBase64(salt),
            }));
        });
    }
    getKeyPair(name) {
        return __awaiter(this, void 0, void 0, function* () {
            const stringified = this.storage.getItem(name);
            if (!stringified) {
                return null;
            }
            try {
                const { encrypted, iv, salt } = JSON.parse(stringified);
                const password = yield this.getDeviceUid();
                const saltArray = this.base64ToUint8(salt);
                const key = yield this.deriveKey(password, saltArray);
                const decrypted = yield this.decryptData(key, encrypted, iv);
                return JSON.parse(decrypted);
            }
            catch (e) {
                console.error('Failed to decrypt key pair:', e);
                return null;
            }
        });
    }
    deriveKey(password, salt) {
        return __awaiter(this, void 0, void 0, function* () {
            const baseKey = yield crypto.subtle.importKey('raw', new TextEncoder().encode(password), { name: DefaultBrowserSecretStorage.KEY_ALGORITHM }, false, ['deriveKey']);
            return crypto.subtle.deriveKey({
                name: DefaultBrowserSecretStorage.KEY_ALGORITHM,
                salt,
                iterations: DefaultBrowserSecretStorage.KEY_ITERATIONS,
                hash: DefaultBrowserSecretStorage.KEY_HASH,
            }, baseKey, { name: DefaultBrowserSecretStorage.AES_ALGORITHM, length: DefaultBrowserSecretStorage.KEY_LENGTH }, false, ['encrypt', 'decrypt']);
        });
    }
    encryptData(key, data) {
        return __awaiter(this, void 0, void 0, function* () {
            const iv = crypto.getRandomValues(new Uint8Array(DefaultBrowserSecretStorage.AES_IV_LENGTH));
            const encrypted = yield crypto.subtle.encrypt({ name: DefaultBrowserSecretStorage.AES_ALGORITHM, iv }, key, new TextEncoder().encode(data));
            return {
                encrypted: this.uint8ToBase64(new Uint8Array(encrypted)),
                iv: this.uint8ToBase64(iv),
            };
        });
    }
    decryptData(key, encrypted, ivBase64) {
        return __awaiter(this, void 0, void 0, function* () {
            const encryptedBuffer = this.base64ToUint8(encrypted).buffer;
            const iv = this.base64ToUint8(ivBase64);
            const decrypted = yield crypto.subtle.decrypt({ name: DefaultBrowserSecretStorage.AES_ALGORITHM, iv }, key, encryptedBuffer);
            return new TextDecoder().decode(decrypted);
        });
    }
    uint8ToBase64(arr) {
        if (typeof Buffer !== 'undefined') {
            return Buffer.from(arr).toString('base64');
        }
        let binary = '';
        for (let i = 0; i < arr.length; i++) {
            binary += String.fromCharCode(arr[i]);
        }
        return btoa(binary);
    }
    base64ToUint8(base64) {
        if (typeof Buffer !== 'undefined') {
            return new Uint8Array(Buffer.from(base64, 'base64'));
        }
        const binary = atob(base64);
        const arr = new Uint8Array(binary.length);
        for (let i = 0; i < binary.length; i++) {
            arr[i] = binary.charCodeAt(i);
        }
        return arr;
    }
}
DefaultBrowserSecretStorage.KEY_ALGORITHM = 'PBKDF2';
DefaultBrowserSecretStorage.KEY_HASH = 'SHA-256';
DefaultBrowserSecretStorage.KEY_LENGTH = 256;
DefaultBrowserSecretStorage.KEY_ITERATIONS = 100000;
DefaultBrowserSecretStorage.AES_ALGORITHM = 'AES-GCM';
DefaultBrowserSecretStorage.AES_IV_LENGTH = 12;
DefaultBrowserSecretStorage.SALT_LENGTH = 16;
exports.default = DefaultBrowserSecretStorage;
//# sourceMappingURL=DefaultBrowserSecretStorage.js.map