"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.showOSDSaga = exports.putOpenOSD = void 0;
const effects_1 = require("redux-saga/effects");
const keyActions_1 = require("@signageos/actions/dist/Input/keyActions");
const socketActionCreator_1 = require("../../Socket/socketActionCreator");
const numericSequence_1 = require("./numericSequence");
const appletTimingActions_1 = require("../AppletTiming/appletTimingActions");
const KeyCode_1 = require("../../NativeDevice/Input/KeyCode");
const appletActions_1 = require("@signageos/actions/dist/Applet/appletActions");
const dependencyInjection_1 = require("../../DI/dependencyInjection");
const FrontCapability_1 = __importDefault(require("../../NativeDevice/Front/FrontCapability"));
const wait_1 = __importDefault(require("@signageos/lib/es6/Timer/wait"));
const OSD_TIME_LIMIT = 10e3;
function* putOpenOSD(frontDriver, state, waitTime) {
    // JS API can call opening OSD multiple times, so it can create internally multiple calls in saga to open OSD.
    // Those requests then can then result as failed, and it will block requests from “first” opened OSD.
    const currentTime = Date.now();
    if (currentTime - state.applet.lastOpenedOSDTime < OSD_TIME_LIMIT) {
        console.warn('OSD is opened too often, skipping.');
        return {};
    }
    yield (0, effects_1.put)({ type: appletTimingActions_1.ShowOSDTime, time: currentTime });
    // This wait fixes the issue when the OSD is opened instantly after the applet is started.
    // On Brightsign it might break calling the JS API, and the OSD is then unresponsive,
    // for that reason we need to wait few seconds.
    yield (0, wait_1.default)(waitTime);
    yield (0, effects_1.put)({
        type: appletTimingActions_1.ShowOSD,
        appletBinaryFile: {
            localUri: frontDriver.getOSDUri(),
        },
    });
    return {};
}
exports.putOpenOSD = putOpenOSD;
exports.showOSDSaga = (0, dependencyInjection_1.withDependencies)(['frontDriver'], function* ({ frontDriver }) {
    const { security } = frontDriver;
    let isCurrentlyOsdShown = false;
    yield (0, effects_1.takeEvery)(appletTimingActions_1.ShowOSD, function* () {
        isCurrentlyOsdShown = true;
    });
    yield (0, socketActionCreator_1.takeEveryAndBindWhenPlatform)(appletActions_1.UpdateActiveAppletBinary, function* () {
        isCurrentlyOsdShown = false;
    });
    yield (0, effects_1.takeEvery)(keyActions_1.HandleKeySequence, function* (handleKeySequence) {
        try {
            const sequence = handleKeySequence.sequence;
            const keySequenceWithoutLastKey = sequence.slice(0, sequence.length - 1);
            const lastKey = sequence[sequence.length - 1];
            const characterSequence = (0, numericSequence_1.getNumericCode)(keySequenceWithoutLastKey);
            const matchesPin = yield security.verifyPin(characterSequence);
            const frontOsdSupported = yield frontDriver.frontSupports(FrontCapability_1.default.FRONT_OSD);
            if (matchesPin && KeyCode_1.KeyCode[lastKey] === KeyCode_1.KeyCode.OK && frontOsdSupported) {
                yield showOSD(frontDriver);
            }
        }
        catch (error) {
            console.error('openOSDSaga', error);
        }
    });
    yield (0, effects_1.takeEvery)(keyActions_1.HandleKeyUp, function* (handleKeyUp) {
        const keyCode = handleKeyUp.keyCode;
        if (isCurrentlyOsdShown && KeyCode_1.KeyCode[keyCode] === KeyCode_1.KeyCode.EXIT) {
            yield hideOSD();
        }
    });
});
function showOSD(frontDriver) {
    const osdUri = frontDriver.getOSDUri();
    return (0, effects_1.put)({
        type: appletTimingActions_1.ShowOSD,
        appletBinaryFile: {
            localUri: osdUri,
        },
    });
}
function hideOSD() {
    return (0, effects_1.put)({ type: appletTimingActions_1.HideOSD });
}
//# sourceMappingURL=osdSagas.js.map