"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.controlActiveAppletTiming = void 0;
const socketActionCreator_1 = require("../../Socket/socketActionCreator");
const effects_1 = require("redux-saga/effects");
const dateTimeFactory_1 = require("@signageos/lib/dist/DateTime/dateTimeFactory");
const appletActions_1 = require("@signageos/actions/dist/Applet/appletActions");
const motionActions_1 = require("@signageos/actions/dist/Input/motionActions");
const keyActions_1 = require("@signageos/actions/dist/Input/keyActions");
const appletTimingActions_1 = require("./appletTimingActions");
const channels_1 = require("../../ReduxSaga/channels");
const deviceConnectActions_1 = require("@signageos/actions/dist/Device/Connect/deviceConnectActions");
const activeAppletActions_1 = require("../Applet/activeAppletActions");
const devicePowerActions_1 = require("@signageos/actions/dist/Device/Power/devicePowerActions");
const PowerActionType_1 = __importDefault(require("@signageos/actions/dist/Device/Power/PowerActionType"));
const generator_1 = require("@signageos/lib/dist/Hash/generator");
const checksum_1 = require("@signageos/lib/dist/Hash/checksum");
const TimingFinishEventType_1 = __importDefault(require("./TimingFinishEventType"));
const deviceConnectActions_2 = require("../../Device/Connect/deviceConnectActions");
function createConnectedAppletTimingDefinition(appletUid, appletVersion) {
    return {
        appletUid,
        configuration: {},
        timmingChecksum: (0, checksum_1.checksumString)(appletUid + ',' + appletVersion),
        appletVersion,
        isPackage: true,
        appletVersionPostfix: (0, generator_1.generateUniqueHash)(),
        finishEvent: {
            type: TimingFinishEventType_1.default[TimingFinishEventType_1.default.DURATION],
            data: null,
        },
        position: 1,
        metadata: {},
    };
}
/**
 * This saga controls the flow of when to run which applet
 *
 * It starts with no definitions and in that case, there are no applets to run so it does nothing.
 *
 * Once it receives action UpdateAppletTimingsDefinition with new timing definitions, it stores the definitions
 * and starts processing them from the beginning.
 *
 * The heart of this process is an infinite loop that iterates over the definitions list and triggers switch to the next applet
 * when conditions are met. However, this loop is not part of this file. Since it's an infinite loop whose only job is to do some
 * simple decision making, it made sense to run it in a separate service worker - i.e. another thread.
 * This infinite loop can be found in src/Front/AppletTiming/AppletTimingController.ts.
 */
function* controlActiveAppletTiming(webWorkerFactory, defaultDefinitions) {
    const defaultDefinitionsOrNull = () => {
        if (defaultDefinitions && defaultDefinitions.length > 0) {
            return defaultDefinitions;
        }
        return null;
    };
    const appletTimingWebWorker = webWorkerFactory.createAppletTimingController();
    const appletTimingChannel = (0, channels_1.createChannel)((putHere) => {
        appletTimingWebWorker.onMessage(putHere);
    });
    appletTimingWebWorker.start(undefined);
    let model;
    const setModel = (newModel) => {
        model = newModel;
        appletTimingWebWorker.postMessage(newModel);
    };
    setModel({
        definitions: defaultDefinitionsOrNull(),
        currentDefinitionIndex: null,
        startTimestamp: null,
        lastInteractionTimestamp: null,
        triggersPausedTimestamp: null,
        lastPutUpdateActiveAppletBinary: null,
        lastPutPrepareNextAppletTiming: null,
        deviceInConnectedMode: null,
    });
    yield (0, socketActionCreator_1.takeEveryAndBindWhenPlatform)(appletActions_1.UpdateAppletTimingsDefinition, function* (action) {
        let definitions;
        if (action.definitions && action.definitions.length > 0) {
            definitions = action.definitions.sort((a, b) => a.position - b.position);
        }
        else {
            definitions = defaultDefinitionsOrNull();
        }
        setModel(Object.assign(Object.assign({}, model), { definitions, currentDefinitionIndex: null, startTimestamp: null, lastInteractionTimestamp: null, triggersPausedTimestamp: null, lastPutUpdateActiveAppletBinary: null, lastPutPrepareNextAppletTiming: null }));
    });
    yield (0, socketActionCreator_1.bindAndTakeEvery)(appletActions_1.LastActiveAppletMissed, function* () {
        const definitions = defaultDefinitionsOrNull();
        setModel(Object.assign(Object.assign({}, model), { definitions, currentDefinitionIndex: null, startTimestamp: null, lastInteractionTimestamp: null, triggersPausedTimestamp: null, lastPutUpdateActiveAppletBinary: null, lastPutPrepareNextAppletTiming: null }));
        if (!defaultDefinitions) {
            yield (0, effects_1.put)({
                type: appletTimingActions_1.HideApplet,
            });
        }
    });
    yield (0, effects_1.takeEvery)([motionActions_1.HandleMotion, keyActions_1.HandleKeyUp], function* () {
        if (model.currentDefinitionIndex !== null && model.triggersPausedTimestamp === null) {
            setModel(Object.assign(Object.assign({}, model), { lastInteractionTimestamp: (0, dateTimeFactory_1.now)().valueOf() }));
        }
    });
    yield (0, effects_1.takeEvery)([appletTimingActions_1.PauseAppletTimingTriggers, appletTimingActions_1.ShowOSD], function* () {
        if (model.currentDefinitionIndex !== null && model.triggersPausedTimestamp === null) {
            setModel(Object.assign(Object.assign({}, model), { triggersPausedTimestamp: (0, dateTimeFactory_1.now)().valueOf() }));
        }
    });
    yield (0, effects_1.takeEvery)([appletTimingActions_1.ResumeAppletTimingTriggers, appletActions_1.UpdateActiveAppletUniqueHash], function* () {
        if (model.startTimestamp !== null && model.triggersPausedTimestamp !== null) {
            const pauseDuration = (0, dateTimeFactory_1.now)().valueOf() - model.triggersPausedTimestamp;
            setModel(Object.assign(Object.assign({}, model), { startTimestamp: model.startTimestamp + pauseDuration, triggersPausedTimestamp: null }));
        }
    });
    yield (0, effects_1.takeEvery)(appletTimingActions_1.HideOSD, function* () {
        setModel(Object.assign(Object.assign({}, model), { currentDefinitionIndex: null, startTimestamp: null, lastInteractionTimestamp: null, triggersPausedTimestamp: null, lastPutUpdateActiveAppletBinary: null, lastPutPrepareNextAppletTiming: null }));
    });
    yield (0, channels_1.takeEvery)(appletTimingChannel, function* () {
        var _a;
        if (model.definitions !== null && model.definitions.length > 0 && !((_a = model.deviceInConnectedMode) === null || _a === void 0 ? void 0 : _a.connected)) {
            const newCurrentIndex = model.currentDefinitionIndex === null ? 0 : (model.currentDefinitionIndex + 1) % model.definitions.length;
            const newNextIndex = (newCurrentIndex + 1) % model.definitions.length;
            setModel(Object.assign(Object.assign({}, model), { currentDefinitionIndex: newCurrentIndex, startTimestamp: (0, dateTimeFactory_1.now)().valueOf(), lastInteractionTimestamp: null, triggersPausedTimestamp: null }));
            const definition = model.definitions[newCurrentIndex];
            yield* putUpdateActiveAppletBinaryIfChanged(definition);
            if (newNextIndex !== newCurrentIndex) {
                const nextDefinition = model.definitions[newNextIndex];
                yield* putPrepareNextAppletTimingIfChanged(nextDefinition);
            }
        }
    });
    yield (0, socketActionCreator_1.takeEveryAndBindWhenPlatform)(deviceConnectActions_2.DeviceConnected, function* (action) {
        setModel(Object.assign(Object.assign({}, model), { deviceInConnectedMode: {
                connected: true,
                appletUid: action.appletUid,
                appletVersion: action.appletVersion,
            } }));
        yield putUpdateActiveAppletBinaryIfChanged(createConnectedAppletTimingDefinition(action.appletUid, action.appletVersion));
    });
    yield (0, socketActionCreator_1.takeEveryAndBindWhenPlatform)(deviceConnectActions_1.DisconnectDevice, function* () {
        setModel(Object.assign(Object.assign({}, model), { deviceInConnectedMode: null }));
        yield (0, effects_1.put)({
            type: devicePowerActions_1.PerformPowerAction,
            powerType: PowerActionType_1.default.APPLET_RELOAD,
        });
    });
    yield (0, effects_1.takeEvery)(activeAppletActions_1.ActiveAppletRestore, function* () {
        var _a;
        if ((_a = model.deviceInConnectedMode) === null || _a === void 0 ? void 0 : _a.connected) {
            yield putUpdateActiveAppletBinaryIfChanged(createConnectedAppletTimingDefinition(model.deviceInConnectedMode.appletUid, model.deviceInConnectedMode.appletVersion));
        }
    });
    function* putUpdateActiveAppletBinaryIfChanged(definition) {
        var _a;
        const action = {
            type: appletActions_1.UpdateActiveAppletBinary,
            activeAppletUid: definition.appletUid,
            activeConfiguration: definition.configuration,
            activeTimingChecksum: definition.timmingChecksum,
            activeAppletVersion: definition.appletVersion,
            activeAppletVersionPostfix: definition.appletVersionPostfix,
            activeAppletFrontAppletVersion: definition.frontAppletVersion,
            activeAppletIsPackage: (_a = definition.isPackage) !== null && _a !== void 0 ? _a : false,
            activeAppletMetadata: definition.metadata,
        };
        if (!model.lastPutUpdateActiveAppletBinary || JSON.stringify(action) !== JSON.stringify(model.lastPutUpdateActiveAppletBinary)) {
            yield (0, effects_1.put)(action);
            setModel(Object.assign(Object.assign({}, model), { lastPutUpdateActiveAppletBinary: action }));
        }
    }
    function* putPrepareNextAppletTimingIfChanged(nextDefinition) {
        var _a;
        const action = {
            type: appletTimingActions_1.PrepareNextAppletTiming,
            appletUid: nextDefinition.appletUid,
            configuration: nextDefinition.configuration,
            timingChecksum: nextDefinition.timmingChecksum,
            appletVersion: nextDefinition.appletVersion,
            appletFrontAppletVersion: nextDefinition.frontAppletVersion,
            isPackage: (_a = nextDefinition.isPackage) !== null && _a !== void 0 ? _a : false,
            appletMetadata: nextDefinition.metadata,
        };
        if (!model.lastPutPrepareNextAppletTiming || JSON.stringify(action) !== JSON.stringify(model.lastPutPrepareNextAppletTiming)) {
            yield (0, effects_1.put)(action);
            setModel(Object.assign(Object.assign({}, model), { lastPutPrepareNextAppletTiming: action }));
        }
    }
}
exports.controlActiveAppletTiming = controlActiveAppletTiming;
//# sourceMappingURL=appletTimingControllerSagas.js.map