"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleSyncMessage = void 0;
const InternalSynchronizerError_1 = __importDefault(require("../Error/InternalSynchronizerError"));
const ErrorCodes_1 = __importDefault(require("../Error/ErrorCodes"));
const appletSyncHandler_utils_1 = require("./appletSyncHandler.utils");
function* handleSyncMessage(messageTypePrefix, data, _nativeDriver, synchronizer, _appletUid, _timingChecksum) {
    switch (data.type) {
        case messageTypePrefix + '.sync.connect':
            return yield handleSyncConnect(data, synchronizer);
        case messageTypePrefix + '.sync.close':
            return yield handleSyncClose(synchronizer);
        // historically this was called 'sync.init' so it has to stay for backwards compatibility
        case messageTypePrefix + '.sync.init':
            return yield handleSyncJoinGroup(data, synchronizer);
        case messageTypePrefix + '.sync.leave_group':
            return yield handleSyncLeaveGroup(data, synchronizer);
        case messageTypePrefix + '.sync.wait':
            return yield handleSyncWait(data, synchronizer);
        case messageTypePrefix + '.sync.cancel_wait':
            return yield handleSyncCancelWait(data, synchronizer);
        // historically this was called 'sync.request_set_value' so it has to stay that way for backwards compatibility
        case messageTypePrefix + '.sync.request_set_value':
            return yield handleBroadcastValue(data, synchronizer);
        case messageTypePrefix + '.sync.is_master':
            return yield handleIsMaster(data, synchronizer);
        default:
            return null;
    }
}
exports.handleSyncMessage = handleSyncMessage;
function* handleSyncConnect({ serverUri, engine }, synchronizer) {
    try {
        const syncType = (0, appletSyncHandler_utils_1.appletSyncEngineToSynchronizerSynchronizerType)(engine);
        yield synchronizer.connect(serverUri, syncType);
    }
    catch (error) {
        throw new InternalSynchronizerError_1.default({
            kind: 'internalSynchronizerErrorWithOrigin',
            message: "Synchronizer couldn't connect to the server.",
            code: ErrorCodes_1.default.SYNCHRONIZER_CONNECT_ERROR,
            origin: 'connect()',
            originStack: error.stack,
            originMessage: error.message,
        });
    }
    return {};
}
function* handleSyncClose(synchronizer) {
    try {
        yield synchronizer.close();
    }
    catch (error) {
        throw new InternalSynchronizerError_1.default({
            kind: 'internalSynchronizerErrorWithOrigin',
            message: "The connection wasn't closed correctly.",
            code: ErrorCodes_1.default.SYNCHRONIZER_CLOSE_ERROR,
            origin: 'close()',
            originStack: error.stack,
            originMessage: error.message,
        });
    }
    return {};
}
function* handleSyncJoinGroup(data, synchronizer) {
    try {
        yield synchronizer.joinGroup({
            groupName: data.groupName,
            deviceIdentification: data.deviceIdentification,
        });
    }
    catch (error) {
        throw new InternalSynchronizerError_1.default({
            kind: 'internalSynchronizerErrorWithOrigin',
            message: 'The group initialization failed.',
            code: ErrorCodes_1.default.SYNCHRONIZER_JOIN_GROUP_ERROR,
            origin: 'joinGroup()',
            originStack: error.stack,
            originMessage: error.message,
        });
    }
    return {};
}
function* handleSyncLeaveGroup(data, synchronizer) {
    try {
        yield synchronizer.leaveGroup(data.groupName);
    }
    catch (error) {
        throw new InternalSynchronizerError_1.default({
            kind: 'internalSynchronizerErrorWithOrigin',
            message: 'Failed to leave the group.',
            code: ErrorCodes_1.default.SYNCHRONIZER_LEAVE_GROUP_ERROR,
            origin: 'leaveGroup()',
            originStack: error.stack,
            originMessage: error.message,
        });
    }
}
function* handleSyncWait(data, synchronizer) {
    try {
        const responseData = yield synchronizer.wait({
            groupName: data.groupName,
            data: data.data,
            timeoutMs: data.timeout,
        });
        return { responseData };
    }
    catch (error) {
        throw new InternalSynchronizerError_1.default({
            kind: 'internalSynchronizerErrorWithOrigin',
            message: 'Wait failed for unexpected reason.',
            code: ErrorCodes_1.default.SYNCHRONIZER_WAIT_ERROR,
            origin: `wait(${data.groupName}, ${data.data}, ${data.timeout})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
}
function* handleSyncCancelWait(data, synchronizer) {
    try {
        yield synchronizer.cancelWait(data.groupName);
        return {};
    }
    catch (error) {
        throw new InternalSynchronizerError_1.default({
            kind: 'internalSynchronizerErrorWithOrigin',
            message: 'Cancel wait failed for unexpected reason.',
            code: ErrorCodes_1.default.SYNCHRONIZER_CANCEL_WAIT_ERROR,
            origin: `cancelWait(${data.groupName})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
}
function* handleBroadcastValue(data, synchronizer) {
    try {
        yield synchronizer.broadcastValue({
            groupName: data.groupName,
            key: data.key,
            value: data.value,
        });
        return {};
    }
    catch (error) {
        throw new InternalSynchronizerError_1.default({
            kind: 'internalSynchronizerErrorWithOrigin',
            message: "Synchronizer couldn't set the value.",
            code: ErrorCodes_1.default.SYNCHRONIZER_BROADCAST_VALUE_ERROR,
            origin: `setValue(${data.groupName}, ${data.key}, ${data.value})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
}
function* handleIsMaster(data, synchronizer) {
    try {
        const isMaster = yield synchronizer.isMaster(data.groupName);
        return { isMaster };
    }
    catch (error) {
        throw new InternalSynchronizerError_1.default({
            kind: 'internalSynchronizerErrorWithOrigin',
            message: "Synchronizer couldn't get the master status.",
            code: ErrorCodes_1.default.SYNCHRONIZER_IS_MASTER_ERROR,
            origin: `isMaster(${data.groupName})`,
            originStack: error.stack,
            originMessage: error.message,
        });
    }
}
function* syncHandler({ messageTypePrefix, data, frontDriver, synchronizer, appletUid, timingChecksum, }) {
    return yield handleSyncMessage(messageTypePrefix, data, frontDriver, synchronizer, appletUid, timingChecksum);
}
exports.default = syncHandler;
//# sourceMappingURL=appletSyncHandler.js.map