"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleOfflineCacheMessage = void 0;
const effects_1 = require("redux-saga/effects");
const HashAlgorithm_1 = __importDefault(require("../../../NativeDevice/HashAlgorithm"));
const AppletNativeCacheError_1 = __importDefault(require("../Error/AppletNativeCacheError"));
const AppletResourceError_1 = __importDefault(require("../Error/AppletResourceError"));
const ErrorCodes_1 = __importDefault(require("../Error/ErrorCodes"));
const ErrorSuggestions_1 = __importDefault(require("../Error/ErrorSuggestions"));
const InternalNativeCacheError_1 = __importDefault(require("../Error/InternalNativeCacheError"));
const InternalOfflineCacheError_1 = __importDefault(require("../Error/InternalOfflineCacheError"));
function getAppletPrefix(appletUid, shortAppletFilesUrl) {
    return `applet/${shortAppletFilesUrl ? appletUid.slice(0, 8) : appletUid}/data/`;
}
function* handleOfflineCacheMessage(messageTypePrefix, data, nativeDriver, offlineCache, appletPrefix, _timingChecksum) {
    switch (data.type) {
        case messageTypePrefix + '.offline.cache.list_files':
            return yield (0, effects_1.call)(handleFileSystemListFiles, offlineCache, appletPrefix);
        case messageTypePrefix + '.offline.cache.list_contents':
            return yield (0, effects_1.call)(handleFileSystemListContents, nativeDriver, appletPrefix);
        case messageTypePrefix + '.offline.cache.save_file':
            return yield (0, effects_1.call)(handleFileSystemSaveFile, data, offlineCache, appletPrefix);
        case messageTypePrefix + '.offline.cache.save_content':
            return yield (0, effects_1.call)(handleFileSystemSaveContent, data, nativeDriver, appletPrefix);
        case messageTypePrefix + '.offline.cache.load_file':
            return yield (0, effects_1.call)(handleFileSystemLoadFile, data, offlineCache, appletPrefix);
        case messageTypePrefix + '.offline.cache.load_content':
            return yield (0, effects_1.call)(handleFileSystemLoadContent, data, nativeDriver, appletPrefix);
        case messageTypePrefix + '.offline.cache.delete_file':
            return yield (0, effects_1.call)(handleFileSystemDeleteFile, data, offlineCache, appletPrefix);
        case messageTypePrefix + '.offline.cache.delete_content':
            return yield (0, effects_1.call)(handleFileSystemDeleteContent, data, nativeDriver, appletPrefix);
        case messageTypePrefix + '.offline.cache.validate_checksum':
            return yield (0, effects_1.call)(handleFileSystemValidateChecksum, data, offlineCache, appletPrefix);
        case messageTypePrefix + '.offline.cache.get_checksum':
            return yield (0, effects_1.call)(handleFileSystemGetChecksum, data, offlineCache, appletPrefix);
        case messageTypePrefix + '.offline.cache.decompress_file':
            return yield (0, effects_1.call)(handleFileSystemDecompressFile, data, offlineCache, appletPrefix);
        default:
            return null;
    }
}
exports.handleOfflineCacheMessage = handleOfflineCacheMessage;
function handleFileSystemListFiles(offlineCache, appletPrefix) {
    return __awaiter(this, void 0, void 0, function* () {
        let fileUids = [];
        try {
            fileUids = yield offlineCache.listFilesRecursively();
        }
        catch (error) {
            throw new InternalOfflineCacheError_1.default({
                kind: 'internalOfflineCacheErrorWithOrigin',
                message: "Couldn't not read the files from the offline cache.",
                code: ErrorCodes_1.default.OFFLINE_CACHE_LIST_FILES_ERROR,
                origin: 'listFilesRecursively',
                originStack: error.stack,
                originMessage: error.message,
            });
        }
        const appletFileUids = [];
        for (let fileUid of fileUids) {
            if (fileUid.indexOf(appletPrefix) === 0) {
                appletFileUids.push(fileUid.substring(appletPrefix.length));
            }
        }
        return {
            fileUids: appletFileUids,
        };
    });
}
function handleFileSystemListContents(nativeDriver, appletPrefix) {
    return __awaiter(this, void 0, void 0, function* () {
        const appletContentUids = [];
        try {
            const contentUids = yield nativeDriver.cacheGetUids();
            for (let contentUid of contentUids) {
                if (contentUid.indexOf(appletPrefix) === 0) {
                    appletContentUids.push(contentUid.substring(appletPrefix.length));
                }
            }
            return {
                contentUids: appletContentUids,
            };
        }
        catch (error) {
            throw new InternalNativeCacheError_1.default({
                kind: 'internalNativeCacheErrorWithOrigin',
                message: 'Unexpected error ocurred when listing content',
                code: ErrorCodes_1.default.NATIVE_CACHE_GET_CONTENT_ERROR,
                origin: 'cacheGetUids',
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
// muttable part
let savingFiles = {};
function handleFileSystemSaveFile(data, offlineCache, appletPrefix) {
    return __awaiter(this, void 0, void 0, function* () {
        const fileUid = appletPrefix + data.uid;
        if (typeof savingFiles[fileUid] !== 'undefined') {
            throw new AppletResourceError_1.default({
                kind: 'appletResourcesError',
                message: 'already loading file: ' + fileUid,
                code: ErrorCodes_1.default.FILE_IS_ALREADY_LOADING,
                suggestion: ErrorSuggestions_1.default.FILE_IS_ALREADY_LOADING,
            });
        }
        const exists = yield offlineCache.fileExists(fileUid);
        if (exists) {
            throw new AppletResourceError_1.default({
                kind: 'appletResourcesError',
                message: 'already existing file: ' + fileUid,
                code: ErrorCodes_1.default.FILE_ALREADY_EXISTS,
                suggestion: ErrorSuggestions_1.default.FILE_ALREADY_EXISTS,
            });
        }
        savingFiles[fileUid] = data;
        console.info('start', fileUid);
        try {
            yield offlineCache.retriableDownloadFile(5, fileUid, data.uri, data.headers);
        }
        finally {
            delete savingFiles[fileUid];
        }
        console.info('end', fileUid);
        const savedFile = yield offlineCache.getFile(fileUid);
        if (savedFile === null) {
            throw new InternalOfflineCacheError_1.default({
                kind: 'internalOfflineCacheError',
                message: "File wasn't saved correctly.",
                code: ErrorCodes_1.default.OFFLINE_CACHE_SAVE_FILE_GET_FILE_ERROR,
            });
        }
        return {
            savedFile: { filePath: savedFile.localUri },
        };
    });
}
// muttable part
let savingContents = {};
function handleFileSystemSaveContent(data, nativeDriver, appletPrefix) {
    return __awaiter(this, void 0, void 0, function* () {
        const contentUid = appletPrefix + data.uid;
        // TODO throw error if content already exists
        if (typeof savingContents[contentUid] !== 'undefined') {
            throw new AppletNativeCacheError_1.default({
                kind: 'appletNativeCacheError',
                message: 'Already saving the file with UID: ' + contentUid,
                code: ErrorCodes_1.default.NATIVE_CACHE_ALREADY_SAVING_CONTENT,
            });
        }
        savingContents[contentUid] = data;
        console.info('start', contentUid);
        try {
            yield nativeDriver.cacheSave(contentUid, data.content);
        }
        catch (error) {
            throw new InternalNativeCacheError_1.default({
                kind: 'internalNativeCacheErrorWithOrigin',
                message: "Couldn't save the file to the offline cache.",
                code: ErrorCodes_1.default.NATIVE_CACHE_SAVE_CONTENT_ERROR,
                origin: `cacheSave(${contentUid}, ${data.content})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
        finally {
            delete savingContents[contentUid];
        }
        console.info('end', contentUid);
        return {
            savedContent: data,
        };
    });
}
function handleFileSystemLoadFile(data, offlineCache, appletPrefix) {
    return __awaiter(this, void 0, void 0, function* () {
        const fileUid = appletPrefix + data.uid;
        const loadedFile = yield offlineCache.getFile(fileUid);
        if (loadedFile === null) {
            throw new InternalOfflineCacheError_1.default({
                kind: 'internalOfflineCacheError',
                message: 'Reading the file from the offline cache failed.',
                code: ErrorCodes_1.default.OFFLINE_CACHE_GET_FILE_ERROR,
            });
        }
        return {
            loadedFile: { filePath: loadedFile.localUri },
        };
    });
}
function handleFileSystemLoadContent(data, nativeDriver, appletPrefix) {
    return __awaiter(this, void 0, void 0, function* () {
        const contentUid = appletPrefix + data.uid;
        try {
            return {
                loadedContent: yield nativeDriver.cacheGet(contentUid),
            };
        }
        catch (error) {
            throw new InternalNativeCacheError_1.default({
                kind: 'internalNativeCacheErrorWithOrigin',
                message: "Couldn't load the file from offline cache.",
                code: ErrorCodes_1.default.NATIVE_CACHE_GET_CONTENT_ERROR,
                origin: `cacheGet(${contentUid})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
    });
}
function handleFileSystemDeleteFile(data, offlineCache, appletPrefix) {
    return __awaiter(this, void 0, void 0, function* () {
        const fileUid = appletPrefix + data.uid;
        yield offlineCache.deleteFileAndDeleteDirectoryIfEmpty(fileUid);
        return {};
    });
}
function handleFileSystemDeleteContent(data, nativeDriver, appletPrefix) {
    return __awaiter(this, void 0, void 0, function* () {
        const contentUid = appletPrefix + data.uid;
        try {
            yield nativeDriver.cacheGet(contentUid); //  if the content doesn't exist, it will reject
        }
        catch (error) {
            throw new InternalNativeCacheError_1.default({
                kind: 'internalNativeCacheErrorWithOrigin',
                message: "Couldn't load the file before deleting it.",
                code: ErrorCodes_1.default.NATIVE_CACHE_GET_CONTENT_ERROR,
                origin: `cacheGet(${contentUid})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
        try {
            yield nativeDriver.cacheDelete(contentUid);
        }
        catch (error) {
            throw new InternalNativeCacheError_1.default({
                kind: 'internalNativeCacheErrorWithOrigin',
                message: "File wasn't deleted correctly.",
                code: ErrorCodes_1.default.NATIVE_CACHE_DELETE_CONTENT_ERROR,
                origin: `cacheDelete(${contentUid})`,
                originStack: error.stack,
                originMessage: error.message,
            });
        }
        return {};
    });
}
function handleFileSystemValidateChecksum(data, offlineCache, appletPrefix) {
    return __awaiter(this, void 0, void 0, function* () {
        const fileUid = appletPrefix + data.uid;
        const exists = yield offlineCache.fileExists(fileUid);
        if (!exists) {
            throw new Error(`file ${fileUid} must be saved before hashing!`);
        }
        const hashAlgorithm = data.hashType;
        if (!isHashAlgorithmSupported(hashAlgorithm)) {
            throw new Error(`hash algorithm ${hashAlgorithm} is not supported`);
        }
        const checksum = yield offlineCache.getFileChecksum(fileUid, hashAlgorithm);
        return { validatedChecksum: checksum === data.hash };
    });
}
function handleFileSystemGetChecksum(data, offlineCache, appletPrefix) {
    return __awaiter(this, void 0, void 0, function* () {
        const fileUid = appletPrefix + data.uid;
        const exists = yield offlineCache.fileExists(fileUid);
        if (!exists) {
            throw new Error(`file ${fileUid} must be saved before hashing!`);
        }
        const hashAlgorithm = data.hashType;
        if (!isHashAlgorithmSupported(hashAlgorithm)) {
            throw new Error(`hash algorithm ${hashAlgorithm} is not supported`);
        }
        const checksum = yield offlineCache.getFileChecksum(fileUid, hashAlgorithm);
        return { checksum };
    });
}
function isHashAlgorithmSupported(hashAlgorithm) {
    return __awaiter(this, void 0, void 0, function* () {
        return Object.values(HashAlgorithm_1.default).includes(hashAlgorithm);
    });
}
function handleFileSystemDecompressFile(data, offlineCache, appletPrefix) {
    return __awaiter(this, void 0, void 0, function* () {
        const fileUid = appletPrefix + data.uid;
        const fileDestinationUid = appletPrefix + data.destinationUid;
        const exists = yield offlineCache.fileExists(fileUid);
        if (!exists) {
            throw new Error(`file ${fileUid} must be saved before decompressing!`);
        }
        yield offlineCache.extractFile(fileUid, fileDestinationUid, data.method);
        return {};
    });
}
function* offlineCacheHandler({ messageTypePrefix, data, cacheDriver, offlineCache, appletUid, timingChecksum, shortAppletFilesUrl, }) {
    const appletPrefix = getAppletPrefix(appletUid, shortAppletFilesUrl);
    return yield (0, effects_1.call)(handleOfflineCacheMessage, messageTypePrefix, data, cacheDriver, offlineCache, appletPrefix, timingChecksum);
}
exports.default = offlineCacheHandler;
//# sourceMappingURL=appletOfflineCacheHandler.js.map