"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.handleBrowserMessage = void 0;
const lodash_1 = __importDefault(require("lodash"));
const effects_1 = require("redux-saga/effects");
const activeAppletActions_1 = require("@signageos/actions/dist/Applet/activeAppletActions");
const AppletBrowserError_1 = __importDefault(require("../Error/AppletBrowserError"));
const ErrorCodes_1 = __importDefault(require("../Error/ErrorCodes"));
const InternalBrowserError_1 = __importDefault(require("../Error/InternalBrowserError"));
function* handleBrowserMessage(messageTypePrefix, data, nativeDriver, appletUid, timingChecksum) {
    var _a, _b;
    switch (data.type) {
        case messageTypePrefix + '.browser.open_link':
        case messageTypePrefix + '.browser.open':
            if (typeof data.uri !== 'string') {
                throw new AppletBrowserError_1.default({
                    kind: 'appletBrowserError',
                    message: 'Property argument uri must be a string.',
                    code: ErrorCodes_1.default.BROWSER_LINK_IS_NOT_STRING,
                });
            }
            yield (0, effects_1.put)({
                type: activeAppletActions_1.ActiveOpenBrowserLink,
                appletUid,
                timingChecksum,
                uri: data.uri,
            });
            try {
                const options = lodash_1.default.omit(data, 'type', 'uri');
                if ('headlessMode' in data && typeof data.headlessMode !== 'undefined') {
                    // New JS API: Populate default values depending on newly passed headlessMode.
                    // Old JS API: Leave implementation without headlessMode up to the platform/application.
                    // Headless mode: Prevent the user from leaving and keep cache around for later.
                    // UI mode: Allow the user to leave and clear cache between sessions.
                    (_a = options.clearData) !== null && _a !== void 0 ? _a : (options.clearData = !data.headlessMode);
                    (_b = options.canUserClose) !== null && _b !== void 0 ? _b : (options.canUserClose = !data.headlessMode);
                }
                if (data.type === messageTypePrefix + '.browser.open_link') {
                    yield nativeDriver.browserOpenLink(data.uri, options);
                }
                else {
                    yield nativeDriver.browser.open(data.uri, options);
                }
            }
            catch (error) {
                throw new InternalBrowserError_1.default({
                    kind: 'internalBrowserErrorWithOrigin',
                    message: `Unexpected error occurred when opening ${data.uri}`,
                    code: ErrorCodes_1.default.BROWSER_OPEN_LINK_ERROR,
                    origin: 'browserOpenLink',
                    originStack: error.stack,
                    originMessage: error.message,
                });
            }
            return {};
        case messageTypePrefix + '.browser.close':
            yield nativeDriver.browser.close();
            return {};
        default:
            return null;
    }
}
exports.handleBrowserMessage = handleBrowserMessage;
function* browserHandler({ messageTypePrefix, data, frontDriver, appletUid, timingChecksum, }) {
    return yield handleBrowserMessage(messageTypePrefix, data, frontDriver, appletUid, timingChecksum);
}
exports.default = browserHandler;
//# sourceMappingURL=appletBrowserHandler.js.map