"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const lodash_1 = __importDefault(require("lodash"));
const checksum_1 = require("../Hash/checksum");
const extractor_1 = require("../Archive/extractor");
const fileSystemHelper_1 = require("./fileSystemHelper");
const basicErrors_1 = require("../NativeDevice/Error/basicErrors");
const debug_1 = __importDefault(require("debug"));
const debug = (0, debug_1.default)('@signageos/front-display:FileSystem:ProprietaryFileSystem');
const METADATA_DIRECTORY = '.metadata';
class ProprietaryFileSystem {
    constructor(window, fileSystemType = window.TEMPORARY, storageSizeBytes = 100 * 1024 * 1024) {
        this.window = window;
        this.fileSystemType = fileSystemType;
        this.storageSizeBytes = storageSizeBytes;
    }
    initialize() {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                yield this.requestFileSystem(this.fileSystemType, this.storageSizeBytes);
            }
            catch (error) {
                console.warn(`Unable to request PERSISTENT file system. Use TEMPORARY instead`, error);
                this.fileSystemType = this.window.TEMPORARY;
                yield this.requestFileSystem(this.fileSystemType, this.storageSizeBytes);
            }
            yield this.listStorageUnits();
        });
    }
    listFiles(directoryPath) {
        return __awaiter(this, void 0, void 0, function* () {
            const fs = yield this.getFileSystem(directoryPath.storageUnit);
            const directoryEntry = yield this.getDirectoryEntry(fs, directoryPath.filePath);
            const directoryReader = directoryEntry.createReader();
            const entries = yield this.readDirectoryEntries(directoryReader);
            const filePaths = entries.map((entry) => {
                return {
                    storageUnit: Object.assign({}, this.storageUnit),
                    filePath: (0, fileSystemHelper_1.trimSlashesAndDots)(`${directoryPath.filePath}/${entry.name}`),
                };
            });
            return filePaths;
        });
    }
    getFile(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const fs = yield this.getFileSystem(filePath.storageUnit);
            try {
                const fileEntry = yield this.getFileEntry(fs, filePath.filePath);
                const localUri = fileEntry.toURL();
                const response = yield this.fetchWithFailoverUsingMetadata(localUri, { method: 'HEAD' });
                let lastModifiedAt = response.headers.has('If-Modified-Since')
                    ? Math.floor(new Date(response.headers.get('If-Modified-Since')).valueOf() / 1000)
                    : undefined;
                lastModifiedAt = response.headers.has('Last-Modified')
                    ? Math.floor(new Date(response.headers.get('Last-Modified')).valueOf() / 1000)
                    : lastModifiedAt;
                const sizeBytes = response.headers.has('Content-Length') ? parseInt(response.headers.get('Content-Length')) : undefined;
                const mimeType = response.headers.has('Content-Type')
                    ? response.headers.get('Content-Type')
                    : yield this.detectMimeTypeOfUri(localUri);
                return {
                    localUri,
                    createdAt: undefined, // it is not detectable
                    lastModifiedAt,
                    sizeBytes,
                    mimeType,
                };
            }
            catch (error) {
                return null;
            }
        });
    }
    readFile(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            return yield this.getFileText(filePath);
        });
    }
    writeFile(filePath, contents) {
        return __awaiter(this, void 0, void 0, function* () {
            const fileTextBase64Encoded = Buffer.from(contents).toString('base64');
            const playlistEncodedUri = `data:application/json;base64,${fileTextBase64Encoded}`;
            yield this.downloadFile(filePath, playlistEncodedUri);
        });
    }
    appendFile(filePath, contents) {
        return __awaiter(this, void 0, void 0, function* () {
            const fileContents = (yield this.exists(filePath)) ? yield this.readFile(filePath) : '';
            const newContents = fileContents + contents;
            yield this.writeFile(filePath, newContents);
        });
    }
    exists(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const fs = yield this.getFileSystem(filePath.storageUnit);
            try {
                yield this.getEntry(fs, filePath.filePath);
                return true;
            }
            catch (error) {
                return false;
            }
        });
    }
    downloadFile(filePath, sourceUri, headers) {
        return __awaiter(this, void 0, void 0, function* () {
            const response = yield this.fetchOrDecodeDataUri(this.window, sourceUri, { headers });
            if (!response.ok) {
                throw new Error(`Not OK response during downloadFile ${sourceUri}`);
            }
            const blob = yield response.blob();
            const fs = yield this.getFileSystem(filePath.storageUnit);
            const fileEntry = yield this.getFileEntry(fs, filePath.filePath, { create: true });
            yield this.writeBlobToFileEntry(fileEntry, blob);
            try {
                const localUri = fileEntry.toURL();
                yield this.setMetadata(localUri, {
                    createdAt: new Date().valueOf(),
                    originalSourceUri: sourceUri,
                    originalRequestHeaders: headers,
                    originalResponseHeaders: this.getHeadersObject(response.headers),
                });
            }
            catch (error) {
                debug('Cannot set metadata', filePath, sourceUri, error);
            }
        });
    }
    uploadFile(filePath, uri, formKey, headers) {
        return __awaiter(this, void 0, void 0, function* () {
            const blob = yield this.getFileBlob(filePath);
            const formData = new FormData();
            formData.append(formKey, blob);
            const response = yield fetch(uri, { method: 'POST', body: formData, headers });
            if (response.ok) {
                try {
                    return yield response.text();
                }
                catch (e) {
                    // ignore if can't read the body (probably form data)
                    return '';
                }
            }
            throw new Error('Error while reading file');
        });
    }
    deleteFile(filePath, recursive) {
        return __awaiter(this, void 0, void 0, function* () {
            if (recursive && (yield this.isDirectory(filePath))) {
                const subFilePaths = yield this.listFiles(filePath);
                yield Promise.all(subFilePaths.map((subFilePath) => this.deleteFile(subFilePath, true)));
            }
            const fs = yield this.getFileSystem(filePath.storageUnit);
            const fileEntry = yield this.getEntry(fs, filePath.filePath);
            yield new Promise((resolve, reject) => {
                fileEntry.remove(() => resolve(), (error) => reject(error));
            });
            try {
                const localUri = fileEntry.toURL();
                yield this.deleteMetadata(localUri);
            }
            catch (error) {
                debug('Cannot delete metadata', filePath, error);
            }
        });
    }
    copyFile(sourceFilePath, destinationFilePath, options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            if (yield this.exists(destinationFilePath)) {
                if (options.overwrite) {
                    yield this.deleteFile(destinationFilePath, true);
                }
                else {
                    throw new Error(`Cannot copy file to existing path ${destinationFilePath}`);
                }
            }
            const sourceFs = yield this.getFileSystem(sourceFilePath.storageUnit);
            const destinationFs = yield this.getFileSystem(destinationFilePath.storageUnit);
            const fileEntry = yield this.getEntry(sourceFs, sourceFilePath.filePath);
            const destinationDirectoryEntry = yield this.getDirectoryEntry(destinationFs, path_1.posix.dirname(destinationFilePath.filePath));
            const destinationFileName = path_1.posix.basename(destinationFilePath.filePath);
            yield new Promise((resolve, reject) => {
                fileEntry.copyTo(destinationDirectoryEntry, destinationFileName, () => resolve(), (error) => reject(error));
            });
            try {
                yield this.copyOrMoveMetadata(fileEntry, destinationFs, destinationFilePath, 'copyTo');
            }
            catch (error) {
                debug('Cannot copy metadata', sourceFilePath, destinationFilePath, error);
            }
        });
    }
    moveFile(sourceFilePath, destinationFilePath, options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            if (yield this.exists(destinationFilePath)) {
                if (options.overwrite) {
                    yield this.deleteFile(destinationFilePath, true);
                }
                else {
                    throw new Error(`Cannot move file to existing path ${destinationFilePath}`);
                }
            }
            const sourceFs = yield this.getFileSystem(sourceFilePath.storageUnit);
            const destinationFs = yield this.getFileSystem(destinationFilePath.storageUnit);
            const fileEntry = yield this.getEntry(sourceFs, sourceFilePath.filePath);
            const destinationDirectoryEntry = yield this.getDirectoryEntry(destinationFs, path_1.posix.dirname(destinationFilePath.filePath));
            const destinationFileName = path_1.posix.basename(destinationFilePath.filePath);
            yield new Promise((resolve, reject) => {
                fileEntry.moveTo(destinationDirectoryEntry, destinationFileName, () => resolve(), (error) => reject(error));
            });
            try {
                yield this.copyOrMoveMetadata(fileEntry, destinationFs, destinationFilePath, 'moveTo');
            }
            catch (error) {
                debug('Cannot move metadata', sourceFilePath, destinationFilePath, error);
            }
        });
    }
    link(_sourceFilePath, _destinationFilePath) {
        throw new basicErrors_1.NotSupportedMethodError('link');
    }
    createDirectory(directoryPath) {
        return __awaiter(this, void 0, void 0, function* () {
            const fs = yield this.getFileSystem(directoryPath.storageUnit);
            yield this.getDirectoryEntry(fs, directoryPath.filePath, { create: true, exclusive: true });
        });
    }
    isDirectory(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const fs = yield this.getFileSystem(filePath.storageUnit);
            const entry = yield this.getEntry(fs, filePath.filePath);
            return entry.isDirectory;
        });
    }
    extractFile(archiveFilePath, destinationDirectoryPath, method) {
        return __awaiter(this, void 0, void 0, function* () {
            if (method !== 'zip') {
                throw new Error(`Unsupported archive method ${method}`);
            }
            const archiveFile = yield this.getFile(archiveFilePath);
            if (!archiveFile) {
                throw new Error(`Archive file ${archiveFilePath.filePath} in storage ${archiveFilePath.storageUnit} doesn't exist.`);
            }
            if (!(yield this.exists(destinationDirectoryPath))) {
                yield this.createDirectory(destinationDirectoryPath);
            }
            const destinationFs = yield this.getFileSystem(destinationDirectoryPath.storageUnit);
            yield (0, extractor_1.extractZipFile)(archiveFile.localUri, (relativeFilePath) => this.ensureDirectory(destinationFs, destinationDirectoryPath.filePath + '/' + relativeFilePath), (relativeFilePath, blob) => __awaiter(this, void 0, void 0, function* () {
                const filePath = destinationDirectoryPath.filePath + '/' + relativeFilePath;
                const fileEntry = yield this.getFileEntry(destinationFs, filePath, { create: true });
                yield this.writeBlobToFileEntry(fileEntry, blob);
            }));
        });
    }
    createArchive(_archiveFilePath, _archiveEntries) {
        return __awaiter(this, void 0, void 0, function* () {
            throw new basicErrors_1.NotSupportedMethodError('createArchive');
        });
    }
    getFileChecksum(filePath, hashType) {
        return __awaiter(this, void 0, void 0, function* () {
            const file = yield this.getFile(filePath);
            if (!file) {
                throw new Error(`File ${filePath.filePath} in storage ${filePath.storageUnit} doesn't exist.`);
            }
            return yield (0, checksum_1.getChecksumOfFile)(this.window, file.localUri, hashType);
        });
    }
    listStorageUnits() {
        return __awaiter(this, void 0, void 0, function* () {
            const storageUnit = this.storageUnit;
            const fs = yield this.getFileSystem(storageUnit);
            const usedSpace = yield this.getDirectoryUsageSize(fs.root);
            storageUnit.freeSpace = storageUnit.usableSpace = storageUnit.capacity - usedSpace;
            return [storageUnit];
        });
    }
    onStorageUnitsChanged(_listener) {
        // Storages never change. Only one available
    }
    getArchiveInfo(archiveFilePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const file = yield this.getFile(archiveFilePath);
            if (!file) {
                throw new Error(`File ${archiveFilePath.filePath} doesn't exist.`);
            }
            const fileData = yield this.fetchWithFailover(file.localUri, { method: 'GET' });
            const uncompressedSize = yield (0, extractor_1.getZipExtractSize)(yield fileData.blob());
            return { uncompressedSize };
        });
    }
    wipeout() {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.deleteFile({ storageUnit: this.storageUnit, filePath: 'data' }, true);
        });
    }
    ensureDirectory(fs, directoryPath) {
        return __awaiter(this, void 0, void 0, function* () {
            const parentDirectoryPath = path_1.posix.dirname(directoryPath);
            if (parentDirectoryPath !== '.') {
                yield this.ensureDirectory(fs, parentDirectoryPath);
            }
            yield this.getDirectoryEntry(fs, directoryPath, { create: true }); // ensure dir
        });
    }
    getDirectoryUsageSize(directoryEntry) {
        return __awaiter(this, void 0, void 0, function* () {
            const directoryReader = directoryEntry.createReader();
            const entries = yield this.readDirectoryEntries(directoryReader);
            const totalUsage = yield entries.reduce((usage, entry) => __awaiter(this, void 0, void 0, function* () {
                if (entry.isDirectory) {
                    return (yield usage) + (yield this.getDirectoryUsageSize(entry));
                }
                else {
                    const fileMetadata = yield this.getFileMatadata(entry);
                    return (yield usage) + fileMetadata.size;
                }
            }), Promise.resolve(0));
            return totalUsage;
        });
    }
    getFileText(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const response = yield this.fetchFile(filePath);
            if (response.ok && response.body !== null) {
                return yield response.text();
            }
            throw new Error('Error while reading file');
        });
    }
    getFileBlob(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const response = yield this.fetchFile(filePath);
            if (response.ok && response.body !== null) {
                return yield response.blob();
            }
            throw new Error('Error while reading file');
        });
    }
    fetchFile(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const fs = yield this.getFileSystem(filePath.storageUnit);
            try {
                const fileEntry = yield this.getFileEntry(fs, filePath.filePath);
                const localUri = fileEntry.toURL();
                return yield this.fetchWithFailoverUsingMetadata(localUri);
            }
            catch (error) {
                throw new Error(`File not found: ${error.message}`);
            }
        });
    }
    getFileSystem(storageUnit) {
        return __awaiter(this, void 0, void 0, function* () {
            if (storageUnit.type !== this.storageUnit.type) {
                throw new Error(`Unknown storage unit type ${storageUnit.type}`);
            }
            if (this.fileSystem) {
                return this.fileSystem;
            }
            const requestFileSystem = this.window.requestFileSystem || this.window.webkitRequestFileSystem;
            this.fileSystem = yield new Promise((resolve, reject) => {
                requestFileSystem(this.fileSystemType, this.storageUnit.capacity, (fileSystem) => resolve(fileSystem), (error) => reject(error));
            });
            return this.fileSystem;
        });
    }
    getEntry(fs, filePath, flags) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                return yield this.getFileEntry(fs, filePath, flags);
            }
            catch (error) {
                return yield this.getDirectoryEntry(fs, filePath, flags);
            }
        });
    }
    getFileEntry(fs, filePath, flags) {
        return new Promise((resolve, reject) => {
            fs.root.getFile(filePath, flags, (entry) => resolve(entry), (error) => reject(error));
        });
    }
    getDirectoryEntry(fs, filePath, flags) {
        return new Promise((resolve, reject) => {
            fs.root.getDirectory(filePath, flags, (entry) => resolve(entry), (error) => reject(error));
        });
    }
    readDirectoryEntries(directoryReader) {
        return new Promise((resolve, reject) => {
            directoryReader.readEntries((entries) => resolve(entries), (error) => reject(error));
        });
    }
    createWriter(fileEntry) {
        return new Promise((resolve, reject) => {
            fileEntry.createWriter((fileWriter) => resolve(fileWriter), (error) => reject(error));
        });
    }
    getFileMatadata(entry) {
        return new Promise((resolve, reject) => {
            entry.getMetadata((metadata) => resolve(metadata), (error) => reject(error));
        });
    }
    writeBlobToFileEntry(fileEntry, blob) {
        return __awaiter(this, void 0, void 0, function* () {
            const fileWriter = yield this.createWriter(fileEntry);
            const truncateEndPromise = new Promise((resolve, reject) => {
                fileWriter.onwriteend = (_event) => resolve();
                fileWriter.onerror = (_event) => reject(new Error(`Error during truncate file ${fileEntry.fullPath} from blob`));
            });
            fileWriter.truncate(0);
            yield truncateEndPromise;
            const writeEndPromise = new Promise((resolve, reject) => {
                fileWriter.onwriteend = (_event) => resolve();
                fileWriter.onerror = (_event) => reject(new Error(`Error during write file ${fileEntry.fullPath} from blob`));
            });
            fileWriter.write(blob);
            yield writeEndPromise;
        });
    }
    detectMimeTypeOfUri(localUri) {
        return __awaiter(this, void 0, void 0, function* () {
            const response = yield this.fetchWithFailoverUsingMetadata(localUri);
            const blob = yield response.blob();
            return blob.type;
        });
    }
    fetchWithFailoverUsingMetadata(localUri, options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            const response = yield this.fetchWithFailover(localUri, options);
            const metadata = yield this.getMetadata(localUri);
            const mixedHeaders = Object.assign(Object.assign(Object.assign({}, metadata === null || metadata === void 0 ? void 0 : metadata.originalResponseHeaders), (metadata ? { 'last-modified': new Date(metadata.createdAt).toString() } : {})), (response.headers ? this.getHeadersObject(response.headers) : {}));
            const overwrittenHeaders = this.getSimpleHeaders(mixedHeaders);
            const extendedResponse = Object.assign(Object.create(Object.getPrototypeOf(response)), response, { headers: overwrittenHeaders });
            return extendedResponse;
        });
    }
    fetchWithFailover(localUri, options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                return yield this.fetchOrDecodeDataUri(this.window, localUri, options);
            }
            catch (error) {
                debug('Original fetch failed', localUri, options, error);
                return yield this.simpleFetch(localUri, options);
            }
        });
    }
    simpleFetch(localUri, options = {}) {
        return new Promise((resolve, reject) => {
            var _a;
            // defaults
            options.method = (_a = options.method) !== null && _a !== void 0 ? _a : 'GET';
            const xhr = new XMLHttpRequest();
            xhr.open(options.method, localUri);
            if (options.method === 'GET') {
                xhr.responseType = 'arraybuffer';
            }
            xhr.onerror = (event) => {
                reject(new Error(`Network request failed for ${localUri}. Loaded ${event.loaded} of ${event.total} bytes.`));
            };
            xhr.onload = () => __awaiter(this, void 0, void 0, function* () {
                const statusGroup = Math.floor(xhr.status / 100);
                const headers = lodash_1.default.fromPairs(xhr
                    .getAllResponseHeaders()
                    .split('\n')
                    .map((header) => {
                    const pair = header.split(/: */);
                    return [pair[0].toLowerCase(), pair[1]];
                }));
                resolve({
                    blob() {
                        return __awaiter(this, void 0, void 0, function* () {
                            const blob = new Blob([new Uint8Array(xhr.response, 0, xhr.response.byteLength)]);
                            return blob;
                        });
                    },
                    text() {
                        return __awaiter(this, void 0, void 0, function* () {
                            const responseUint16Array = new Uint16Array(xhr.response);
                            const responseNumberArray = Array.from(responseUint16Array);
                            return String.fromCharCode.apply(null, responseNumberArray);
                        });
                    },
                    ok: statusGroup === 2 || statusGroup === 3,
                    body: xhr.response,
                    headers: this.getSimpleHeaders(headers),
                });
            });
            xhr.send();
        });
    }
    getMetadata(localUri) {
        return __awaiter(this, void 0, void 0, function* () {
            const fs = yield this.getFileSystem(this.storageUnit);
            const metadataFilePath = this.getMetadataFilePath(localUri);
            try {
                const fileEntry = yield this.getFileEntry(fs, metadataFilePath);
                const metadataLocalUri = fileEntry.toURL();
                const metadataResponse = yield this.fetchOrDecodeDataUri(window, metadataLocalUri);
                if (metadataResponse.ok && metadataResponse.body !== null) {
                    return yield metadataResponse.json();
                }
            }
            catch (error) {
                debug(`Failed to read metadata`, localUri, error);
                return null;
            }
            return null;
        });
    }
    setMetadata(localUri, metadata) {
        return __awaiter(this, void 0, void 0, function* () {
            const fileTextBase64Encoded = Buffer.from(JSON.stringify(metadata)).toString('base64');
            const playlistEncodedUri = `data:application/json;base64,${fileTextBase64Encoded}`;
            const response = yield this.fetchOrDecodeDataUri(window, playlistEncodedUri);
            const blob = yield response.blob();
            const fs = yield this.getFileSystem(this.storageUnit);
            yield this.getDirectoryEntry(fs, METADATA_DIRECTORY, { create: true }); // ensure metadata dir
            const metadataFilePath = this.getMetadataFilePath(localUri);
            const fileEntry = yield this.getFileEntry(fs, metadataFilePath, { create: true });
            yield this.writeBlobToFileEntry(fileEntry, blob);
        });
    }
    copyOrMoveMetadata(originalFileEntry, destinationFs, destinationFilePath, method) {
        return __awaiter(this, void 0, void 0, function* () {
            const fs = yield this.getFileSystem(this.storageUnit);
            const metadataSourceFileEntry = yield this.getEntry(fs, this.getMetadataFilePath(originalFileEntry.toURL()));
            const metadataDirectoryEntry = yield this.getDirectoryEntry(fs, METADATA_DIRECTORY);
            const destinationFileEntry = yield this.getFileEntry(destinationFs, destinationFilePath.filePath);
            const metadataNewFileName = path_1.posix.basename(this.getMetadataFilePath(destinationFileEntry.toURL()));
            yield new Promise((resolve, reject) => {
                metadataSourceFileEntry[method](metadataDirectoryEntry, metadataNewFileName, () => resolve(), (error) => reject(error));
            });
        });
    }
    deleteMetadata(localUri) {
        return __awaiter(this, void 0, void 0, function* () {
            const fs = yield this.getFileSystem(this.storageUnit);
            const fileEntry = yield this.getEntry(fs, this.getMetadataFilePath(localUri));
            yield new Promise((resolve, reject) => {
                fileEntry.remove(() => resolve(), (error) => reject(error));
            });
        });
    }
    getMetadataFilePath(localUri) {
        const metadataFilePath = `${METADATA_DIRECTORY}/${(0, checksum_1.checksumString)(localUri)}`;
        return metadataFilePath;
    }
    getSimpleHeaders(headersObject) {
        return {
            has(name) {
                return typeof headersObject[name.toLowerCase()] !== 'undefined';
            },
            get(name) {
                return headersObject[name.toLowerCase()];
            },
            forEach(callbackfn) {
                for (const key in headersObject) {
                    callbackfn(headersObject[key], key);
                }
            },
        };
    }
    getHeadersObject(headers) {
        const headersObject = {};
        headers.forEach((value, key) => (headersObject[key.toLowerCase()] = value));
        return headersObject;
    }
    requestFileSystem(fileSystemType, storageSizeBytes) {
        return __awaiter(this, void 0, void 0, function* () {
            let storageInfo;
            if (fileSystemType === this.window.PERSISTENT) {
                storageInfo = this.window.navigator.webkitPersistentStorage;
            }
            else if (fileSystemType === this.window.TEMPORARY) {
                storageInfo = this.window.navigator.webkitTemporaryStorage;
            }
            else {
                console.warn(`Unknown FS type ${fileSystemType}`);
                return 0;
            }
            let grantedSizeBytes = yield new Promise((resolve, reject) => {
                try {
                    storageInfo.queryUsageAndQuota((usedBytes, availableBytes) => resolve(usedBytes + availableBytes), (error) => reject(error));
                }
                catch (error) {
                    reject(error);
                }
            });
            if (grantedSizeBytes === 0) {
                grantedSizeBytes = yield new Promise((resolve, reject) => {
                    try {
                        storageInfo.requestQuota(storageSizeBytes, (grantedBytes) => resolve(grantedBytes), (error) => reject(error));
                    }
                    catch (error) {
                        reject(error);
                    }
                });
            }
            if (storageSizeBytes !== grantedSizeBytes) {
                debug(`Requested for ${storageSizeBytes} bytes of FS but got only ${grantedSizeBytes} bytes`);
            }
            this.storageUnit = {
                type: fileSystemType.toString(),
                capacity: grantedSizeBytes,
                freeSpace: 0,
                usableSpace: 0,
                removable: false,
            };
            yield this.getFileSystem(this.storageUnit); // Just check FS works
        });
    }
    // XMLHttpRequest (inside of fetch polyfill) can't handle datauris
    fetchOrDecodeDataUri(window, ...[req, init]) {
        return __awaiter(this, void 0, void 0, function* () {
            if (typeof req !== 'string' || !req.startsWith('data:')) {
                return yield window.fetch(req, init);
            }
            else {
                const url = new URL(req);
                let [, data] = url.pathname.split(';');
                if (data.startsWith('base64,')) {
                    data = atob(data.slice('base64,'.length));
                }
                const blob = new Blob([data]);
                return new Response(blob);
            }
        });
    }
}
exports.default = ProprietaryFileSystem;
//# sourceMappingURL=ProprietaryFileSystem.js.map