"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DATABASE_PREFIX = void 0;
const path_1 = __importDefault(require("path"));
const basicErrors_1 = require("../../NativeDevice/Error/basicErrors");
const debug_1 = __importDefault(require("debug"));
const eventsHelper_1 = require("../../Events/eventsHelper");
const generator_1 = require("@signageos/lib/dist/Hash/generator");
const errors_1 = require("../errors");
const fetch_1 = require("../../polyfills/fetch");
const extractor_1 = require("../../Archive/extractor");
const checksum_1 = require("../../Hash/checksum");
const lockedDecorator_1 = require("@signageos/lib/es6/Lock/lockedDecorator");
/** The Cache API needs to have native instance of Response however it can be overwritten with fetch polyfill */
const Response = (0, fetch_1.getNativeResponse)();
const debug = (0, debug_1.default)('@signageos/front-display:FileSystem:IndexedDBFileSystem');
const HARD_LIMIT_BYTES = 4 * 1024 * 1024 * 1024; // 4GB
function MemoryAddress(address) {
    return (address !== null && address !== void 0 ? address : (0, generator_1.generateUniqueHash)());
}
var TransactionMode;
(function (TransactionMode) {
    TransactionMode["ReadOnly"] = "readonly";
    TransactionMode["ReadWrite"] = "readwrite";
})(TransactionMode || (TransactionMode = {}));
var Index;
(function (Index) {
    Index["ParentFilePath"] = "parentFilePath";
})(Index || (Index = {}));
var FileType;
(function (FileType) {
    FileType["File"] = "file";
    FileType["Directory"] = "directory";
})(FileType || (FileType = {}));
const INTERNAL_TYPE = 'internal';
const MEMORY_TYPE_PREFIX = '_memory_';
const LOCAL_PATH_PREFIX = '/indexed_db/';
exports.DATABASE_PREFIX = 'fileSystem';
class IndexedDBFileSystem {
    constructor(indexedDB, caches, baseUrl, namespace) {
        this.indexedDB = indexedDB;
        this.caches = caches;
        this.baseUrl = baseUrl;
        this.namespace = namespace;
    }
    initialize() {
        return __awaiter(this, void 0, void 0, function* () {
            this.databaseName = `${exports.DATABASE_PREFIX}${this.namespace ? '_' + this.namespace : ''}`;
            debug('initialize', this.databaseName, this.indexedDB);
            const cache = yield this.caches.open(this.databaseName);
            this.cache = cache;
            const request = this.indexedDB.open(this.databaseName, 1);
            const upgradeEvent = yield (0, eventsHelper_1.waitForSuccessEventsOrFailEvents)(request, ['success', 'upgradeneeded'], ['error']);
            if (upgradeEvent.type === 'upgradeneeded' && upgradeEvent.target) {
                debug('upgradeneeded', upgradeEvent);
                const upgradeRequest = upgradeEvent.target;
                this.db = upgradeRequest.result;
                yield this.ensureInternalStorageUnit();
                if (upgradeRequest.transaction) {
                    upgradeRequest.transaction.commit();
                    yield (0, eventsHelper_1.waitForSuccessEventsOrFailEvents)(upgradeRequest.transaction, ['complete'], []);
                }
            }
            else {
                debug('success', upgradeEvent);
                this.db = request.result;
            }
        });
    }
    destroy() {
        return __awaiter(this, void 0, void 0, function* () {
            this.db.close();
        });
    }
    listFiles(directoryPath) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!(yield this.isDirectory(directoryPath))) {
                throw new errors_1.FileIsNotListableError(directoryPath);
            }
            const { objectStore, commit } = this.startTransaction(directoryPath.storageUnit, { mode: TransactionMode.ReadOnly }, 'listFiles', directoryPath.filePath);
            const keys = yield this.doRequest(objectStore.index(Index.ParentFilePath).getAllKeys(directoryPath.filePath));
            commit();
            if (!keys) {
                throw new errors_1.InvalidFileStateError(directoryPath);
            }
            const filePaths = keys.map((filePath) => ({ storageUnit: directoryPath.storageUnit, filePath }));
            return filePaths;
        });
    }
    getFile(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            try {
                const { blob } = yield this.readFileAsBlob(filePath);
                const localUri = this.createLocalUriWithBaseOrigin(filePath);
                const file = {
                    localUri,
                    mimeType: blob.type,
                    sizeBytes: blob.size,
                };
                return file;
            }
            catch (error) {
                return null;
            }
        });
    }
    readFile(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const { blob } = yield this.readFileAsBlob(filePath);
            const contents = yield blob.text();
            return contents;
        });
    }
    writeFile(filePath, contents) {
        return __awaiter(this, void 0, void 0, function* () {
            const byteArray = Buffer.from(contents, 'utf8');
            const blob = new Blob([byteArray], { type: 'text/plain' });
            yield this.writeBlobAsFile(filePath, blob);
        });
    }
    appendFile(filePath, contents) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!(yield this.exists(filePath))) {
                yield this.writeFile(filePath, contents);
            }
            else {
                const parentDirectoryPath = this.getParentFilePath(filePath);
                if (!(yield this.exists(parentDirectoryPath))) {
                    throw new errors_1.FileDoesNotExistError(parentDirectoryPath);
                }
                const byteArray = Buffer.from(contents, 'utf8');
                const appendBlob = new Blob([byteArray], { type: 'text/plain' });
                try {
                    const { blob: existingBlob, fileMetadata } = yield this.readFileAsBlob(filePath);
                    const blob = new Blob([existingBlob, appendBlob], { type: existingBlob.type });
                    const { commit, memoryObjectStore } = this.startTransaction(filePath.storageUnit, { mode: TransactionMode.ReadWrite, memory: true }, 'appendFile', filePath.filePath);
                    yield this.doRequest(memoryObjectStore.put(blob, fileMetadata.address));
                    commit();
                    yield this.cache.put(this.createLocalUri(filePath), new Response(blob));
                }
                catch (error) {
                    if (error instanceof errors_1.DirectoryIsNotReadableError) {
                        throw new errors_1.DirectoryIsNotWritableError(filePath);
                    }
                    else {
                        throw error;
                    }
                }
            }
        });
    }
    exists(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const { objectStore, commit } = this.startTransaction(filePath.storageUnit, { mode: TransactionMode.ReadOnly }, 'exists', filePath.filePath);
            const key = yield this.doRequest(objectStore.getKey(filePath.filePath));
            commit();
            return key !== undefined;
        });
    }
    downloadFile(filePath, sourceUri, headers) {
        return __awaiter(this, void 0, void 0, function* () {
            const response = yield fetch(sourceUri, { headers });
            if (!response.ok) {
                throw new Error(`Not OK response during downloadFile ${sourceUri}`);
            }
            const blob = yield response.blob();
            yield this.writeBlobAsFile(filePath, blob);
        });
    }
    uploadFile(filePath, uri, formKey, headers) {
        return __awaiter(this, void 0, void 0, function* () {
            const { blob } = yield this.readFileAsBlob(filePath);
            const formData = new FormData();
            formData.append(formKey, blob);
            const response = yield fetch(uri, { method: 'POST', body: formData, headers });
            if (response.ok) {
                try {
                    return yield response.text();
                }
                catch (e) {
                    // ignore if can't read the body (probably form data)
                    return '';
                }
            }
            throw new Error('Error while reading file');
        });
    }
    deleteFile(filePath, recursive) {
        return __awaiter(this, void 0, void 0, function* () {
            if (filePath.filePath === '') {
                throw new errors_1.RootDirectoryIsNotDeletableError(filePath);
            }
            if (!(yield this.exists(filePath))) {
                throw new errors_1.FileDoesNotExistError(filePath);
            }
            if (!recursive) {
                if ((yield this.isDirectory(filePath)) && (yield this.listFiles(filePath)).length > 0) {
                    throw new errors_1.DirectoryIsNotEmptyError(filePath);
                }
            }
            const deleteFileRecursive = (deleteKey) => __awaiter(this, void 0, void 0, function* () {
                const { objectStore, memoryObjectStore, commit, abort } = this.startTransaction(filePath.storageUnit, { mode: TransactionMode.ReadWrite, memory: true }, 'deleteFile', filePath.filePath);
                const fileMetadata = yield this.doRequest(objectStore.get(deleteKey));
                const deleteFilePath = { storageUnit: filePath.storageUnit, filePath: deleteKey };
                if (!fileMetadata) {
                    abort();
                    throw new errors_1.FileDoesNotExistError(deleteFilePath);
                }
                if (fileMetadata.fileType === FileType.Directory) {
                    const [keys] = yield Promise.all([
                        this.doRequest(objectStore.index(Index.ParentFilePath).getAllKeys(deleteKey)),
                        this.doRequest(objectStore.delete(deleteKey)),
                    ]);
                    commit();
                    if (keys) {
                        yield Promise.all(keys.map((key) => deleteFileRecursive(key)));
                    }
                }
                else {
                    yield Promise.all([this.doRequest(memoryObjectStore.delete(fileMetadata.address)), this.doRequest(objectStore.delete(deleteKey))]);
                    commit();
                }
                yield this.cache.delete(this.createLocalUri(deleteFilePath), { ignoreSearch: true, ignoreMethod: true });
            });
            yield deleteFileRecursive(filePath.filePath);
        });
    }
    copyFile(sourceFilePath, destinationFilePath, options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!options.overwrite && (yield this.exists(destinationFilePath))) {
                throw new errors_1.FileAlreadyExistsError(destinationFilePath);
            }
            const destinationParentFilePath = this.getParentFilePath(destinationFilePath);
            if (!(yield this.exists(destinationParentFilePath))) {
                throw new errors_1.FileDoesNotExistError(destinationParentFilePath);
            }
            if (yield this.isDirectory(sourceFilePath)) {
                yield this.ensureDirectoryRecursive(destinationFilePath);
                const sourceSubFilePaths = yield this.listFiles(sourceFilePath);
                yield Promise.all(sourceSubFilePaths.map((sourceSubFilePath) => __awaiter(this, void 0, void 0, function* () {
                    const destSubFilePath = {
                        storageUnit: destinationFilePath.storageUnit,
                        filePath: destinationFilePath.filePath + '/' + path_1.default.basename(sourceSubFilePath.filePath),
                    };
                    yield this.copyFile(sourceSubFilePath, destSubFilePath, options);
                })));
            }
            else {
                const { blob } = yield this.readFileAsBlob(sourceFilePath);
                if ((yield this.exists(destinationFilePath)) && (yield this.isDirectory(destinationFilePath))) {
                    yield this.deleteFile(destinationFilePath, true);
                }
                yield this.writeBlobAsFile(destinationFilePath, blob);
            }
        });
    }
    moveFile(sourceFilePath, destinationFilePath, options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.copyFile(sourceFilePath, destinationFilePath, options);
            yield this.deleteFile(sourceFilePath, true);
        });
    }
    link(_sourceFilePath, _destinationFilePath) {
        throw new basicErrors_1.NotSupportedMethodError('link');
    }
    createDirectory(directoryPath) {
        return __awaiter(this, void 0, void 0, function* () {
            const exists = yield this.exists(directoryPath);
            if (exists) {
                throw new errors_1.FileAlreadyExistsError(directoryPath);
            }
            const parentDirectoryPath = this.getParentFilePath(directoryPath);
            if (!(yield this.exists(parentDirectoryPath))) {
                throw new errors_1.FileDoesNotExistError(parentDirectoryPath);
            }
            if (!(yield this.isDirectory(parentDirectoryPath))) {
                throw new errors_1.FileIsNotListableError(parentDirectoryPath);
            }
            const { objectStore, commit } = this.startTransaction(directoryPath.storageUnit, { mode: TransactionMode.ReadWrite }, 'createDirectory', directoryPath.filePath);
            const fileMetadata = {
                parentFilePath: this.getParentFilePath(directoryPath).filePath,
                fileType: FileType.Directory,
            };
            yield this.doRequest(objectStore.put(fileMetadata, directoryPath.filePath));
            commit();
        });
    }
    isDirectory(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const { objectStore, commit } = this.startTransaction(filePath.storageUnit, { mode: TransactionMode.ReadOnly }, 'isDirectory', filePath.filePath);
            const fileMetadata = yield this.doRequest(objectStore.get(filePath.filePath));
            commit();
            if (!fileMetadata) {
                throw new errors_1.FileDoesNotExistError(filePath);
            }
            return fileMetadata.fileType === FileType.Directory;
        });
    }
    extractFile(archiveFilePath, destinationDirectoryPath, method) {
        return __awaiter(this, void 0, void 0, function* () {
            if (method !== 'zip') {
                throw new Error(`Unsupported archive method ${method}`);
            }
            const { blob } = yield this.readFileAsBlob(archiveFilePath);
            if (!(yield this.exists(destinationDirectoryPath))) {
                yield this.createDirectory(destinationDirectoryPath);
            }
            yield (0, extractor_1.extractZipBlob)(blob, (relativeFilePath) => this.ensureDirectoryRecursive({
                storageUnit: destinationDirectoryPath.storageUnit,
                filePath: destinationDirectoryPath.filePath + '/' + relativeFilePath,
            }), (relativeFilePath, fileBlob) => this.writeBlobAsFile({
                storageUnit: destinationDirectoryPath.storageUnit,
                filePath: destinationDirectoryPath.filePath + '/' + relativeFilePath,
            }, fileBlob));
        });
    }
    createArchive(_archiveFilePath, _archiveEntries) {
        return __awaiter(this, void 0, void 0, function* () {
            throw new basicErrors_1.NotSupportedMethodError('createArchive');
        });
    }
    getFileChecksum(filePath, hashType) {
        return __awaiter(this, void 0, void 0, function* () {
            const { blob } = yield this.readFileAsBlob(filePath);
            return yield (0, checksum_1.getChecksumOfBlob)(blob, hashType);
        });
    }
    listStorageUnits() {
        return __awaiter(this, void 0, void 0, function* () {
            const storageUnits = [];
            for (let i = 0; i < this.db.objectStoreNames.length; i++) {
                const type = this.db.objectStoreNames[i];
                if (!type.startsWith(MEMORY_TYPE_PREFIX)) {
                    storageUnits.push({
                        type,
                        removable: type !== INTERNAL_TYPE,
                        // TODO compute sizes based on real data
                        capacity: HARD_LIMIT_BYTES,
                        freeSpace: HARD_LIMIT_BYTES,
                        usableSpace: HARD_LIMIT_BYTES,
                    });
                }
            }
            debug('listStorageUnits', storageUnits);
            return storageUnits;
        });
    }
    onStorageUnitsChanged(_listener) {
        // Storages never change. Only one available
    }
    getArchiveInfo(archiveFilePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const { blob } = yield this.readFileAsBlob(archiveFilePath);
            const uncompressedSize = yield (0, extractor_1.getZipExtractSize)(blob);
            return { uncompressedSize };
        });
    }
    readFileAsBlob(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            const { objectStore, memoryObjectStore, commit, abort } = this.startTransaction(filePath.storageUnit, { mode: TransactionMode.ReadOnly, memory: true }, 'readFileAsBlob', filePath.filePath);
            const fileMetadata = yield this.doRequest(objectStore.get(filePath.filePath));
            if (!fileMetadata) {
                abort();
                throw new errors_1.FileDoesNotExistError(filePath);
            }
            if (fileMetadata.fileType === FileType.Directory) {
                abort();
                throw new errors_1.DirectoryIsNotReadableError(filePath);
            }
            const blob = yield this.doRequest(memoryObjectStore.get(fileMetadata.address));
            if (!blob) {
                abort();
                throw new errors_1.InvalidFileStateError(filePath);
            }
            commit();
            return { blob, fileMetadata };
        });
    }
    writeBlobAsFile(filePath, blob) {
        return __awaiter(this, void 0, void 0, function* () {
            const exists = yield this.exists(filePath);
            if (exists && (yield this.isDirectory(filePath))) {
                throw new errors_1.DirectoryIsNotWritableError(filePath);
            }
            const parentDirectoryPath = this.getParentFilePath(filePath);
            if (!(yield this.exists(parentDirectoryPath))) {
                throw new errors_1.FileDoesNotExistError(parentDirectoryPath);
            }
            const { objectStore, memoryObjectStore, commit } = this.startTransaction(filePath.storageUnit, { mode: TransactionMode.ReadWrite, memory: true }, 'writeBlobAsFile', filePath.filePath);
            if (exists) {
                const existingFileMetadata = yield this.doRequest(objectStore.get(filePath.filePath));
                if (existingFileMetadata) {
                    yield this.doRequest(memoryObjectStore.delete(existingFileMetadata.address));
                }
            }
            const address = MemoryAddress();
            const fileMetadata = {
                parentFilePath: this.getParentFilePath(filePath).filePath,
                fileType: FileType.File,
                address,
            };
            yield Promise.all([
                this.doRequest(memoryObjectStore.put(blob, address)),
                this.doRequest(objectStore.put(fileMetadata, filePath.filePath)),
            ]);
            commit();
            yield this.cache.put(this.createLocalUri(filePath), new Response(blob));
        });
    }
    wipeout() {
        return __awaiter(this, void 0, void 0, function* () {
            const clearCachePromise = new Promise((resolve) => __awaiter(this, void 0, void 0, function* () {
                for (let key of yield this.caches.keys()) {
                    yield this.caches.delete(key);
                }
                resolve();
            }));
            this.destroy();
            yield Promise.all([this.doRequest(this.indexedDB.deleteDatabase(this.databaseName)), clearCachePromise]);
        });
    }
    ensureDirectoryRecursive(directoryPath) {
        return __awaiter(this, void 0, void 0, function* () {
            if (!(yield this.exists(directoryPath))) {
                yield this.ensureDirectoryRecursive(this.getParentFilePath(directoryPath));
                yield this.createDirectory(directoryPath);
            }
        });
    }
    doRequest(request) {
        return __awaiter(this, void 0, void 0, function* () {
            const event = yield (0, eventsHelper_1.waitForSuccessEventsOrFailEvents)(request, ['success'], ['error']);
            return event.target.result;
        });
    }
    startTransaction(storageUnit, options, ...messages) {
        const storeNames = [storageUnit.type];
        if (options.memory) {
            storeNames.push(this.getMemoryType(storageUnit.type));
        }
        debug('transaction start', ...messages);
        const tx = this.db.transaction(storeNames, options.mode);
        tx.addEventListener('complete', (event) => debug('transaction complete', ...messages, event.target));
        const objectStore = tx.objectStore(storageUnit.type);
        let memoryObjectStore;
        if (options.memory) {
            const memoryStorageType = this.getMemoryType(storageUnit.type);
            memoryObjectStore = tx.objectStore(memoryStorageType);
        }
        return {
            tx,
            objectStore,
            memoryObjectStore,
            commit() {
                try {
                    tx.commit();
                    debug('transaction commit succeeded', ...messages);
                }
                catch (error) {
                    debug('transaction commit failed', ...messages, error);
                    throw error;
                }
            },
            abort() {
                try {
                    tx.abort();
                    debug('transaction abort succeeded', ...messages);
                }
                catch (error) {
                    debug('transaction abort failed', ...messages, error);
                    throw error;
                }
            },
        };
    }
    ensureInternalStorageUnit() {
        return __awaiter(this, void 0, void 0, function* () {
            if (!this.db.objectStoreNames.contains(INTERNAL_TYPE)) {
                const objectStore = this.db.createObjectStore(INTERNAL_TYPE);
                objectStore.createIndex(Index.ParentFilePath, 'parentFilePath');
                const fileMetadata = {
                    parentFilePath: null,
                    fileType: FileType.Directory,
                };
                yield this.doRequest(objectStore.put(fileMetadata, ''));
            }
            const memoryInternalType = this.getMemoryType(INTERNAL_TYPE);
            if (!this.db.objectStoreNames.contains(memoryInternalType)) {
                this.db.createObjectStore(memoryInternalType);
            }
        });
    }
    getMemoryType(type) {
        return `${MEMORY_TYPE_PREFIX}${type}`;
    }
    getParentFilePath(filePath) {
        let parentPath = path_1.default.dirname(filePath.filePath);
        if (parentPath === '.') {
            parentPath = '';
        }
        return {
            storageUnit: filePath.storageUnit,
            filePath: parentPath,
        };
    }
    createLocalUri(filePath) {
        const namespace = this.namespace ? this.namespace + '/' : '';
        return `${LOCAL_PATH_PREFIX}${namespace}${filePath.storageUnit.type}/${filePath.filePath}`;
    }
    createLocalUriWithBaseOrigin(filePath) {
        const namespace = this.namespace ? this.namespace + '/' : '';
        const baseUrl = this.baseUrl.endsWith('/') ? this.baseUrl.substring(0, this.baseUrl.length - 1) : this.baseUrl;
        return `${baseUrl}${LOCAL_PATH_PREFIX}${namespace}${filePath.storageUnit.type}/${filePath.filePath}`;
    }
}
exports.default = IndexedDBFileSystem;
__decorate([
    (0, lockedDecorator_1.locked)(function (directoryPath) {
        return this.createLocalUri(directoryPath);
    }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], IndexedDBFileSystem.prototype, "ensureDirectoryRecursive", null);
//# sourceMappingURL=IndexedDBFileSystem.js.map