"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.awaitDependencies = exports.withDependencies = exports.injectDependencies = void 0;
const lodash_1 = require("lodash");
const effects_1 = require("redux-saga/effects");
const dependenciesActions_1 = require("./dependenciesActions");
/**
 * Use this function any time the dynamic dependencies are changed.
 */
function* injectDependencies(dependencies) {
    yield (0, effects_1.put)({
        type: dependenciesActions_1.UpdateDIDependency,
        dependencies,
    });
}
exports.injectDependencies = injectDependencies;
/**
 * withDependencies will not call the wrapped saga. If you'd like to process or yield saga you have to call it.
 * E.g.:
 *
 * ```ts
 * const mySaga = withDependencies(['dep1'], function* ({ dep1 }) {});
 * yield mySaga();
 * // or
 * yield call(mySaga);
 * ```
 *
 * The `();` at the end is necessary otherwise nothing is happening and saga is frozen.
 */
const withDependencies = (dependencyKeys, saga) => function* (...args) {
    let runningTask;
    let lastContainer;
    let action;
    yield (0, effects_1.put)({
        type: dependenciesActions_1.RequestDIContainer,
    });
    while ((action = yield (0, effects_1.take)(dependenciesActions_1.UpdateDIContainer))) {
        const newContainer = (0, lodash_1.pick)(action.container, dependencyKeys);
        if (!isShallowEqual(lastContainer, newContainer) && hasAllRequiredDependencies(dependencyKeys, newContainer)) {
            lastContainer = newContainer;
            if (runningTask) {
                yield (0, effects_1.cancel)(runningTask);
            }
            runningTask = yield (0, effects_1.fork)(saga, newContainer, ...args);
        }
    }
};
exports.withDependencies = withDependencies;
/**
 * If you want to get dependencies as soon as they will be available.
 */
function* awaitDependencies(dependencyKeys) {
    let action;
    yield (0, effects_1.put)({
        type: dependenciesActions_1.RequestDIContainer,
    });
    while ((action = yield (0, effects_1.take)(dependenciesActions_1.UpdateDIContainer))) {
        if (hasAllRequiredDependencies(dependencyKeys, action.container)) {
            return (0, lodash_1.pick)(action.container, dependencyKeys);
        }
    }
}
exports.awaitDependencies = awaitDependencies;
function hasAllRequiredDependencies(dependencyKeys, container) {
    return dependencyKeys.every((key) => Object.keys(container).includes(key));
}
function isShallowEqual(a, b) {
    return (0, lodash_1.isEqualWith)(a, b, (aVal, bBal, index) => (index === undefined ? undefined : Object.is(aVal, bBal)));
}
//# sourceMappingURL=dependencyInjection.js.map