"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const monitoringCommands_1 = require("./monitoringCommands");
const timingCommands_1 = require("./Timing/timingCommands");
const htmlCommands_1 = require("./Html/htmlCommands");
const consoleCommands_1 = require("./Console/consoleCommands");
const offlineCacheCommands_1 = require("./Offline/Cache/offlineCacheCommands");
const videoCommands_1 = require("./Video/videoCommands");
const fileSystemCommands_1 = require("./FileSystem/fileSystemCommands");
class Monitoring {
    constructor(frontApplet) {
        this.frontApplet = frontApplet;
        this.frontApplet.command.onCommand((commandEvent) => {
            if (commandEvent.command.type === monitoringCommands_1.EnableMonitoring) {
                this.frontApplet.config.sosMonitoring = true;
            }
        });
        this.frontApplet.command.onCommand((commandEvent) => {
            if (commandEvent.command.type === monitoringCommands_1.DisableMonitoring) {
                this.frontApplet.config.sosMonitoring = false;
            }
        });
        this.handleTimingLoaded();
        this.handleHtmlSnapshotTakingRequests();
        this.interceptConsoleLogging();
        this.handleOfflineCacheFilesRequests();
        this.handleVideoStatesChanges();
        this.handleFileSystemRequests();
    }
    onCommand(listener) {
        this.frontApplet.command.onCommand((command) => {
            if (typeof this.frontApplet.config === 'object' && this.frontApplet.config.sosMonitoring) {
                listener(command);
            }
        });
    }
    dispatch(command) {
        if (typeof this.frontApplet.config === 'object' && this.frontApplet.config.sosMonitoring) {
            this.frontApplet.command.dispatch(command);
        }
    }
    handleTimingLoaded() {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.frontApplet.onReady();
            this.dispatch({ type: timingCommands_1.TimingLoaded });
        });
    }
    handleHtmlSnapshotTakingRequests() {
        this.onCommand((commandEvent) => {
            switch (commandEvent.command.type) {
                case htmlCommands_1.TakeHtmlSnapshot:
                    this.dispatch({
                        type: htmlCommands_1.HtmlSnapshotTaken,
                        html: this.frontApplet.window.document.documentElement.outerHTML,
                    });
                    break;
                default:
            }
        });
    }
    interceptConsoleLogging() {
        const consoleLog = console.log;
        console.log = (...messages) => {
            consoleLog.apply(console, messages);
            this.dispatch({
                type: consoleCommands_1.ConsoleLogged,
                level: 'log',
                messages,
            });
        };
        const consoleError = console.error;
        console.error = (...messages) => {
            consoleError.apply(console, messages);
            this.dispatch({
                type: consoleCommands_1.ConsoleLogged,
                level: 'error',
                messages,
            });
        };
        const consoleWarn = console.warn;
        console.warn = (...messages) => {
            consoleWarn.apply(console, messages);
            this.dispatch({
                type: consoleCommands_1.ConsoleLogged,
                level: 'warn',
                messages,
            });
        };
        const consoleInfo = console.info;
        console.info = (...messages) => {
            consoleInfo.apply(console, messages);
            this.dispatch({
                type: consoleCommands_1.ConsoleLogged,
                level: 'info',
                messages,
            });
        };
        const consoleDebug = console.debug;
        console.debug = (...messages) => {
            consoleDebug.apply(console, messages);
            this.dispatch({
                type: consoleCommands_1.ConsoleLogged,
                level: 'debug',
                messages,
            });
        };
    }
    handleFileSystemRequests() {
        this.onCommand((commandEvent) => __awaiter(this, void 0, void 0, function* () {
            switch (commandEvent.command.type) {
                case fileSystemCommands_1.FileSystemListOfStorageUnitsRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemListOfStorageUnitsResult,
                        storageUnits: yield this.frontApplet.fileSystem.listStorageUnits(),
                    });
                    break;
                case fileSystemCommands_1.FileSystemListFilesRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemListFilesResult,
                        pathList: yield this.frontApplet.fileSystem.listFiles(commandEvent.command.filePath),
                    });
                    break;
                case fileSystemCommands_1.FileSystemGetFileRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemGetFileResult,
                        file: yield this.frontApplet.fileSystem.getFile(commandEvent.command.filePath),
                    });
                    break;
                case fileSystemCommands_1.FileSystemExistsRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemExistsResult,
                        result: yield this.frontApplet.fileSystem.exists(commandEvent.command.filePath),
                    });
                    break;
                case fileSystemCommands_1.FileSystemDownloadFileRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemDownloadFileResult,
                        result: yield this.frontApplet.fileSystem.downloadFile(commandEvent.command.filePath, commandEvent.command.sourceUri, commandEvent.command.headers),
                    });
                    break;
                case fileSystemCommands_1.FileSystemDeleteFileRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemDeleteFileResult,
                        result: yield this.frontApplet.fileSystem.deleteFile(commandEvent.command.filePath, commandEvent.command.recursive),
                    });
                    break;
                case fileSystemCommands_1.FileSystemMoveFileRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemMoveFileResult,
                        result: yield this.frontApplet.fileSystem.moveFile(commandEvent.command.sourceFilePath, commandEvent.command.destinationFilePath, commandEvent.command.options),
                    });
                    break;
                case fileSystemCommands_1.FileSystemGetFileChecksumRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemGetFileChecksumResult,
                        result: yield this.frontApplet.fileSystem.getFileChecksum(commandEvent.command.filePath, commandEvent.command.hashType),
                    });
                    break;
                case fileSystemCommands_1.FileSystemExtractFileRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemExtractFileResult,
                        result: yield this.frontApplet.fileSystem.extractFile(commandEvent.command.archiveFilePath, commandEvent.command.destinationDirectionPath, commandEvent.command.method),
                    });
                    break;
                case fileSystemCommands_1.FileSystemCreateDirectoryRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemCreateDirectoryResult,
                        result: yield this.frontApplet.fileSystem.createDirectory(commandEvent.command.directoryPath),
                    });
                    break;
                case fileSystemCommands_1.FileSystemCreateArchiveFileRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemCreateArchiveFileResult,
                        result: yield this.frontApplet.fileSystem.createArchive(commandEvent.command.archiveFilePath, commandEvent.command.archiveEntries),
                    });
                    break;
                case fileSystemCommands_1.FileSystemIsDirectoryRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemIsDirectoryResult,
                        result: yield this.frontApplet.fileSystem.isDirectory(commandEvent.command.filePath),
                    });
                    break;
                case fileSystemCommands_1.FileSystemCopyFileRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemCopyFileResult,
                        result: yield this.frontApplet.fileSystem.copyFile(commandEvent.command.sourceFilePath, commandEvent.command.destinationFilePath, commandEvent.command.options),
                    });
                    break;
                case fileSystemCommands_1.FileSystemWriteFileRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemWriteFileResult,
                        result: yield this.frontApplet.fileSystem.writeFile(commandEvent.command.filePath, commandEvent.command.content),
                    });
                    break;
                case fileSystemCommands_1.FileSystemAppendFileRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemAppendFileResult,
                        result: yield this.frontApplet.fileSystem.appendFile(commandEvent.command.filePath, commandEvent.command.content),
                    });
                    break;
                case fileSystemCommands_1.FileSystemReadFileRequest:
                    this.dispatch({
                        type: fileSystemCommands_1.FileSystemReadFileResult,
                        data: yield this.frontApplet.fileSystem.readFile(commandEvent.command.filePath),
                    });
                    break;
                default:
            }
        }));
    }
    handleOfflineCacheFilesRequests() {
        this.onCommand((commandEvent) => __awaiter(this, void 0, void 0, function* () {
            switch (commandEvent.command.type) {
                case offlineCacheCommands_1.OfflineCacheListFiles:
                    this.dispatch({
                        type: offlineCacheCommands_1.OfflineCacheFilesListed,
                        fileUids: yield this.frontApplet.offline.cache.listFiles(),
                    });
                    break;
                case offlineCacheCommands_1.OfflineCacheLoadFile:
                    this.dispatch({
                        type: offlineCacheCommands_1.OfflineCacheFileLoaded,
                        file: yield this.frontApplet.offline.cache.loadFile(commandEvent.command.uid),
                    });
                    break;
                case offlineCacheCommands_1.OfflineCacheGetChecksumRequest:
                    this.dispatch({
                        type: offlineCacheCommands_1.OfflineCacheGetChecksumResult,
                        result: yield this.frontApplet.offline.cache.getChecksumFile(commandEvent.command.uid, commandEvent.command.hashType),
                    });
                    break;
                case offlineCacheCommands_1.OfflineCacheValidateChecksumRequest:
                    this.dispatch({
                        type: offlineCacheCommands_1.OfflineCacheValidateChecksumResult,
                        result: yield this.frontApplet.offline.cache.validateChecksumFile(commandEvent.command.uid, commandEvent.command.hash, commandEvent.command.hashType),
                    });
                    break;
                case offlineCacheCommands_1.OfflineCacheLoadOrSaveFileRequest:
                    this.dispatch({
                        type: offlineCacheCommands_1.OfflineCacheLoadOrSaveFileResult,
                        file: yield this.frontApplet.offline.cache.loadOrSaveFile(commandEvent.command.uid, commandEvent.command.uri, commandEvent.command.headers),
                    });
                    break;
                default:
            }
        }));
    }
    handleVideoStatesChanges() {
        this.frontApplet.video.onPlay((event) => {
            this.dispatch(Object.assign({ type: videoCommands_1.VideoStateChanged, state: 'play' }, event.srcArguments));
        });
        this.frontApplet.video.onPrepare((event) => {
            this.dispatch(Object.assign({ type: videoCommands_1.VideoStateChanged, state: 'prepare' }, event.srcArguments));
        });
        this.frontApplet.video.onStop((event) => {
            this.dispatch(Object.assign({ type: videoCommands_1.VideoStateChanged, state: 'stop' }, event.srcArguments));
        });
        this.frontApplet.video.onPause((event) => {
            this.dispatch(Object.assign({ type: videoCommands_1.VideoStateChanged, state: 'pause' }, event.srcArguments));
        });
        this.frontApplet.video.onResume((event) => {
            this.dispatch(Object.assign({ type: videoCommands_1.VideoStateChanged, state: 'resume' }, event.srcArguments));
        });
        this.frontApplet.video.onEnded((event) => {
            this.dispatch(Object.assign({ type: videoCommands_1.VideoStateChanged, state: 'ended' }, event.srcArguments));
        });
        this.frontApplet.video.onError((event) => {
            this.dispatch(Object.assign({ type: videoCommands_1.VideoStateChanged, state: 'error' }, event.srcArguments));
        });
    }
}
exports.default = Monitoring;
//# sourceMappingURL=Montoring.js.map