"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const FileNotFoundError_1 = require("../../Error/FileNotFoundError");
const ErrorCodes_1 = require("../../Error/ErrorCodes");
const AppletOfflineCacheError_1 = require("../../Error/AppletOfflineCacheError");
const Validate_1 = require("../../Validate/Validate");
class Cache {
    constructor(messagePrefix, postMessage) {
        this.messagePrefix = messagePrefix;
        this.postMessage = postMessage;
        this.savedFiles = {};
        this.savedContents = {};
    }
    listFiles() {
        return __awaiter(this, void 0, void 0, function* () {
            const response = yield this.postMessage({
                type: this.getMessage('list_files'),
            });
            return response.fileUids;
        });
    }
    loadFile(uid) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ uid }).required().string();
            const response = yield this.postMessage({
                type: this.getMessage('load_file'),
                uid,
            });
            this.savedFiles[uid] = response.loadedFile;
            return response.loadedFile;
        });
    }
    saveFile(uid, uri, headers) {
        return __awaiter(this, void 0, void 0, function* () {
            this.checkUidValidity(uid);
            if (headers && !this.isHeaders(headers)) {
                throw new AppletOfflineCacheError_1.default({
                    kind: 'appletOfflineCacheError',
                    message: `Invalid headers ${headers}`,
                    code: ErrorCodes_1.default.OFFLINE_CACHE_HEADERS_NOT_VALID,
                });
            }
            const response = yield this.postMessage({
                type: this.getMessage('save_file'),
                uid,
                uri,
                headers,
            });
            this.savedFiles[uid] = response.savedFile;
        });
    }
    loadOrSaveFile(uid, uri, headers) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ uid }).required().string();
            Validate_1.default({ uri }).required().uri();
            try {
                return this.getCachedFile(uid);
            }
            catch (error) {
                try {
                    return yield this.loadFile(uid);
                }
                catch (error) {
                    yield this.saveFile(uid, uri, headers);
                    return this.getCachedFile(uid);
                }
            }
        });
    }
    deleteFile(uid) {
        return __awaiter(this, void 0, void 0, function* () {
            this.checkUidValidity(uid);
            yield this.postMessage({
                type: this.getMessage('delete_file'),
                uid,
            });
            delete this.savedFiles[uid];
        });
    }
    listContents() {
        return __awaiter(this, void 0, void 0, function* () {
            const response = yield this.postMessage({
                type: this.getMessage('list_contents'),
            });
            return response.contentUids;
        });
    }
    loadContent(uid) {
        return __awaiter(this, void 0, void 0, function* () {
            const response = yield this.postMessage({
                type: this.getMessage('load_content'),
                uid,
            });
            this.savedContents[uid] = response.loadedContent;
            return response.loadedContent;
        });
    }
    saveContent(uid, content) {
        return __awaiter(this, void 0, void 0, function* () {
            this.checkUidValidity(uid);
            Validate_1.default({ content }).required().string();
            const response = yield this.postMessage({
                type: this.getMessage('save_content'),
                uid,
                content: '' + content,
            });
            this.savedContents[uid] = response.savedContent;
        });
    }
    deleteContent(uid) {
        return __awaiter(this, void 0, void 0, function* () {
            this.checkUidValidity(uid);
            yield this.postMessage({
                type: this.getMessage('delete_content'),
                uid,
            });
            delete this.savedContents[uid];
        });
    }
    validateChecksumFile(uid, hash, hashType) {
        return __awaiter(this, void 0, void 0, function* () {
            this.checkUidValidity(uid);
            Validate_1.default({ hash }).required().string();
            Validate_1.default({ hashType }).required().string();
            const response = yield this.postMessage({
                type: this.getMessage('validate_checksum'),
                uid,
                hash,
                hashType,
            });
            return response.validatedChecksum;
        });
    }
    getChecksumFile(uid, hashType) {
        return __awaiter(this, void 0, void 0, function* () {
            this.checkUidValidity(uid);
            Validate_1.default({ hashType }).required().string();
            const response = yield this.postMessage({
                type: this.getMessage('get_checksum'),
                uid,
                hashType,
            });
            return response.checksum;
        });
    }
    decompressFile(uid, destinationUid, method) {
        return __awaiter(this, void 0, void 0, function* () {
            this.checkUidValidity(uid);
            Validate_1.default({ destinationUid }).required().string();
            yield this.postMessage({
                type: this.getMessage('decompress_file'),
                uid,
                destinationUid,
                method,
            });
        });
    }
    getCachedFile(uid) {
        if (typeof this.savedFiles[uid] === 'undefined') {
            throw new FileNotFoundError_1.default({
                kind: 'fileNotFoundError',
                message: 'File was not found ' + uid,
                code: ErrorCodes_1.default.FILE_NOT_FOUND_IN_CACHE,
            });
        }
        return this.savedFiles[uid];
    }
    getMessage(name) {
        return this.messagePrefix + '.' + Cache.MESSAGE_PREFIX + '.' + name;
    }
    checkUidValidity(uid) {
        const regex = /^[^\\:*"<>|]+$/g; // Allow all characters allowed by all OSes in fileName + slash /
        Validate_1.default({ uid }).required().string().matchRegExp(regex, ErrorCodes_1.default.OFFLINE_CACHE_UID_NOT_VALID);
        return true;
    }
    isHeaders(headers) {
        if (typeof headers !== 'object') {
            return false;
        }
        Object.getOwnPropertyNames(headers).forEach(function (val) {
            if (typeof val !== 'string' || typeof headers[val] !== 'string') {
                return false;
            }
        });
        return true;
    }
}
exports.default = Cache;
Cache.MESSAGE_PREFIX = 'cache';
//# sourceMappingURL=Cache.js.map