"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const events_1 = require("events");
const generator_1 = require("./Hash/generator");
const Video_1 = require("./Video/Video");
const Stream_1 = require("./Stream/Stream");
const Offline_1 = require("./Offline/Offline");
const JavascriptLoadFileController_1 = require("./Offline/LoadFile/JavascriptLoadFileController");
const Command_1 = require("./Command/Command");
const Hardware_1 = require("./Hardware/Hardware");
const Browser_1 = require("./Browser/Browser");
const Iframe_1 = require("./Iframe/Iframe");
const Touch_1 = require("./Touch/Touch");
const Input_1 = require("./Input/Input");
const Debug_1 = require("./Debug/Debug");
const DeviceInfo_1 = require("./DeviceInfo/DeviceInfo");
const Display_1 = require("./Display/Display");
const Timing_1 = require("./Timing/Timing");
const Sync_1 = require("./Sync/Sync");
const FileSystem_1 = require("./FileSystem/FileSystem");
const errorHelper_1 = require("./Error/errorHelper");
const Management_1 = require("./Management/Management");
const Monitors_1 = require("./Monitors/Monitors");
const Sensors_1 = require("./Sensors/Sensors");
const Exchange_1 = require("./Exchange/Exchange");
const OSD_1 = require("./OSD/OSD");
const ProofOfPlay_1 = require("./ProofOfPlay/ProofOfPlay");
class FrontApplet {
    constructor(window, messagePrefix) {
        this.window = window;
        this.messagePrefix = messagePrefix;
        const rootExchange = Exchange_1.createExchange(messagePrefix, this.postMessage.bind(this));
        this.readyEmitter = new events_1.EventEmitter();
        this.invocationResponseEmitter = new events_1.EventEmitter();
        this.debug = new Debug_1.default(this.window, messagePrefix, this.postMessage.bind(this));
        this.deviceInfo = new DeviceInfo_1.default(messagePrefix, this.postMessage.bind(this));
        this.display = new Display_1.default(messagePrefix, this.postMessage.bind(this));
        this.stream = new Stream_1.default(messagePrefix, this.postMessage.bind(this));
        this.video = new Video_1.default(messagePrefix, this.postMessage.bind(this));
        this.command = new Command_1.default(this.messagePrefix, this.postMessage.bind(this));
        this.hardware = new Hardware_1.default(this.messagePrefix, this.postMessage.bind(this));
        this.offline = new Offline_1.default(messagePrefix, this.window, this.postMessage.bind(this));
        this.browser = new Browser_1.default(rootExchange.createNamespaced('browser'));
        this.touch = new Touch_1.default(this.window, this.messagePrefix, this.postMessage.bind(this));
        this.iframe = new Iframe_1.default(this.window, this.messagePrefix, this, [this.touch.transformIframeData.bind(this.touch)]);
        this.input = new Input_1.default(this.window, this.messagePrefix);
        this.timing = new Timing_1.default(this.messagePrefix, this.postMessage.bind(this));
        this.sync = new Sync_1.default(this.messagePrefix, this.postMessage.bind(this));
        this.fileSystem = new FileSystem_1.default(this.messagePrefix, this.postMessage.bind(this));
        this.management = new Management_1.default(messagePrefix, this.postMessage.bind(this));
        this.monitors = new Monitors_1.default(messagePrefix, this.postMessage.bind(this));
        this.osd = new OSD_1.default(messagePrefix, this.postMessage.bind(this));
        this.proofOfPlay = new ProofOfPlay_1.default(this.command);
        this.sensors = new Sensors_1.default(messagePrefix);
        this.javascriptLoadFileController = new JavascriptLoadFileController_1.default(this.window);
        this.window.addEventListener('message', (event) => this.handleMessage(event));
        this.postMessage({
            type: messagePrefix + '_api.ready',
        });
        this.startPinging();
    }
    postMessage(data) {
        const invocationUid = generator_1.generateUniqueHash();
        this.window.setTimeout(() => {
            var _a;
            return void ((_a = this.window.parent) === null || _a === void 0 ? void 0 : _a.postMessage(Object.assign({ invocationUid }, data), '*'));
        });
        return new Promise((resolve, reject) => {
            this.invocationResponseEmitter.once(invocationUid, (responseData) => {
                if (responseData.error) {
                    if (responseData.error.error) {
                        reject(errorHelper_1.createError(responseData.error.error));
                    }
                    else {
                        reject(responseData.error);
                    }
                }
                else {
                    resolve(responseData.success);
                }
            });
        });
    }
    onReady(listener) {
        return new Promise((resolve) => {
            this.readyEmitter.once('ready', () => {
                resolve();
                if (typeof listener !== 'undefined') {
                    listener();
                }
            });
            this.checkIfReady();
        });
    }
    restore() {
        this.postMessage({ type: this.getMessage('restore') });
        this.removeMediaListeners();
        this.forwardRestoreToFrontAppletWindows();
    }
    refresh() {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.postMessage({
                type: this.getMessage('applet_refresh'),
            });
        });
    }
    forwardRestoreToFrontAppletWindows() {
        const data = {
            type: this.messagePrefix + 'restore',
        };
        this.iframe.forwardToFrontAppletWindows(data);
    }
    removeMediaListeners() {
        this.video.removeEventListeners();
        this.stream.removeEventListeners();
        this.sync.removeEventListeners();
        this.input.removeEventListeners();
    }
    checkIfReady() {
        if (typeof this.authHash !== 'undefined' &&
            typeof this.config !== 'undefined' &&
            this.display.version !== null) {
            this.readyEmitter.emit('ready');
        }
    }
    handleMessage(event) {
        if (typeof event.data === 'object') {
            const source = event.source;
            if (source === this.window.parent) {
                const data = event.data;
                switch (data.type) {
                    case this.getMessage('invocation.success'):
                        this.invocationResponseEmitter.emit(data.invocationUid, { success: data });
                        break;
                    case this.getMessage('invocation.error'):
                        this.invocationResponseEmitter.emit(data.invocationUid, { error: data });
                        break;
                    case this.getMessage('config.update'):
                        this.config = data.config;
                        this.checkIfReady();
                        break;
                    case this.getMessage('auth_hash.update'):
                        this.authHash = data.authHash;
                        this.checkIfReady();
                        break;
                    case this.getMessage('front_display_version.update'):
                        this.display.version = data.version;
                        this.checkIfReady();
                        break;
                    case this.getMessage('applet_version.update'):
                        this.appletVersion = data.version;
                        break;
                    case this.getMessage('restore'):
                        this.removeMediaListeners();
                        break;
                    case this.getMessage('assets.load_js'):
                        this.javascriptLoadFileController.load(data.src, this.window.document.body);
                        break;
                    default:
                        this.browser.handleMessageData(data);
                        this.sync.handleMessageData(data);
                        this.video.handleMessageData(data);
                        this.stream.handleMessageData(data);
                        this.input.handleMessageData(data);
                        this.command.handleMessageData(data);
                        this.fileSystem.handleMessageData(data);
                        this.sensors.handleMessageData(data);
                        this.management.wifi.handleMessageData(data);
                        this.hardware.handleMessageData(data);
                        this.touch.handleMessageData(data);
                        this.debug.handleMessageData(data);
                }
                this.iframe.forwardToFrontAppletWindows(data);
            }
            else if (isWindow(source) && source !== this.window) {
                const data = event.data;
                switch (data.type) {
                    case this.messagePrefix + '_loader.ready':
                        this.iframe.registerFrontAppletLoaderWindow(source);
                        break;
                    case this.messagePrefix + '_api.ready':
                        this.iframe.registerBundledFrontAppletWindow(source);
                        break;
                    default:
                        this.iframe.forwardFromFrontAppletWindows(source, data);
                }
            }
        }
    }
    getMessage(name) {
        return this.messagePrefix + '.' + name;
    }
    startPinging() {
        const doPing = () => this.postMessage({ type: this.getMessage('ping') });
        this.window.setInterval(doPing, 30e3);
        doPing();
    }
}
exports.default = FrontApplet;
function isWindow(source) {
    return Boolean(source);
}
//# sourceMappingURL=FrontApplet.js.map