"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const events_1 = require("events");
const types_1 = require("./types");
const Validate_1 = require("../Validate/Validate");
const EVENT_STORAGE_UNITS_CHANGED = 'storage_units_changed';
class FileSystem {
    constructor(messagePrefix, postMessage) {
        this.messagePrefix = messagePrefix;
        this.postMessage = postMessage;
        this.eventEmitter = new events_1.EventEmitter();
    }
    listStorageUnits() {
        return __awaiter(this, void 0, void 0, function* () {
            const { storageUnits } = yield this.postMessage({
                type: this.getMessage('list_storage_units'),
            });
            return storageUnits;
        });
    }
    onStorageUnitsChanged(listener) {
        Validate_1.default({ listener }).required().function();
        this.eventEmitter.addListener(EVENT_STORAGE_UNITS_CHANGED, listener);
    }
    removeStorageUnitsChangedListener(listener) {
        Validate_1.default({ listener }).required().function();
        this.eventEmitter.removeListener(EVENT_STORAGE_UNITS_CHANGED, listener);
    }
    removeAllListeners() {
        this.eventEmitter.removeAllListeners(EVENT_STORAGE_UNITS_CHANGED);
    }
    listFiles(directoryPath) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ directoryPath }).required().object(types_1.VIFilePath);
            const { files } = yield this.postMessage({
                type: this.getMessage('list_files'),
                directoryPath,
            });
            return files;
        });
    }
    exists(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ filePath }).required().object(types_1.VIFilePath);
            const { exists } = yield this.postMessage({
                type: this.getMessage('exists'),
                filePath,
            });
            return exists;
        });
    }
    getFile(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ filePath }).required().object(types_1.VIFilePath);
            const { file } = yield this.postMessage({
                type: this.getMessage('get_file'),
                filePath,
            });
            return file;
        });
    }
    writeFile(filePath, contents) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ filePath }).required().object(types_1.VIFilePath);
            yield this.postMessage({
                type: this.getMessage('write_file'),
                filePath,
                contents,
            });
        });
    }
    appendFile(filePath, contents) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ filePath }).required().object(types_1.VIFilePath);
            yield this.postMessage({
                type: this.getMessage('append_file'),
                filePath,
                contents,
            });
        });
    }
    readFile(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ filePath }).required().object(types_1.VIFilePath);
            const { contents } = yield this.postMessage({
                type: this.getMessage('read_file'),
                filePath,
            });
            return contents;
        });
    }
    copyFile(sourceFilePath, destinationFilePath, options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ sourceFilePath }).required().object(types_1.VIFilePath);
            Validate_1.default({ destinationFilePath }).required().object(types_1.VIFilePath);
            Validate_1.default({ options }).required().object(types_1.VICopyFileOptions);
            yield this.postMessage({
                type: this.getMessage('copy_file'),
                sourceFilePath,
                destinationFilePath,
                options,
            });
        });
    }
    moveFile(sourceFilePath, destinationFilePath, options = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ sourceFilePath }).required().object(types_1.VIFilePath);
            Validate_1.default({ destinationFilePath }).required().object(types_1.VIFilePath);
            Validate_1.default({ options }).required().object(types_1.VIMoveFileOptions);
            yield this.postMessage({
                type: this.getMessage('move_file'),
                sourceFilePath,
                destinationFilePath,
                options,
            });
        });
    }
    deleteFile(filePath, recursive) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ filePath }).required().object(types_1.VIFilePath);
            Validate_1.default({ recursive }).required().boolean();
            yield this.postMessage({
                type: this.getMessage('delete_file'),
                filePath,
                recursive,
            });
        });
    }
    downloadFile(filePath, sourceUri, headers) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ filePath }).required().object(types_1.VIFilePath);
            Validate_1.default({ sourceUri }).required().uri();
            Validate_1.default({ headers }).object();
            yield this.postMessage({
                type: this.getMessage('download_file'),
                filePath,
                sourceUri,
                headers,
            });
        });
    }
    extractFile(archiveFilePath, destinationDirectoryPath, method) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ archiveFilePath }).required().object(types_1.VIFilePath);
            Validate_1.default({ destinationDirectoryPath }).object(types_1.VIFilePath);
            Validate_1.default({ method }).required().string();
            yield this.postMessage({
                type: this.getMessage('extract_file'),
                archiveFilePath,
                destinationDirectoryPath,
                method,
            });
        });
    }
    createArchive(archiveFilePath, archiveEntries) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ archiveFilePath }).required().object(types_1.VIFilePath);
            Validate_1.default({ archiveEntries }).required().array('object');
            yield this.postMessage({
                type: this.getMessage('create_archive_file'),
                archiveFilePath,
                archiveEntries,
            });
        });
    }
    getFileChecksum(filePath, hashType) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ filePath }).required().object(types_1.VIFilePath);
            Validate_1.default({ hashType }).required().string();
            const { checksum } = yield this.postMessage({
                type: this.getMessage('get_file_checksum'),
                filePath,
                hashType,
            });
            return checksum;
        });
    }
    createDirectory(directoryPath) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ directoryPath }).required().object(types_1.VIFilePath);
            yield this.postMessage({
                type: this.getMessage('create_directory'),
                directoryPath,
            });
        });
    }
    isDirectory(filePath) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ filePath }).required().object(types_1.VIFilePath);
            const { isDirectory } = yield this.postMessage({
                type: this.getMessage('is_directory'),
                filePath,
            });
            return isDirectory;
        });
    }
    link(sourceFilePath, destinationFilePath) {
        return __awaiter(this, void 0, void 0, function* () {
            Validate_1.default({ sourceFilePath }).required().object(types_1.VIFilePath);
            Validate_1.default({ destinationFilePath }).required().object(types_1.VIFilePath);
            yield this.postMessage({
                type: this.getMessage('link'),
                sourceFilePath,
                destinationFilePath,
            });
        });
    }
    handleMessageData(data) {
        switch (data.type) {
            case this.getMessage('storage_units_changed'):
                this.eventEmitter.emit(EVENT_STORAGE_UNITS_CHANGED);
                break;
            default:
        }
    }
    getMessage(name) {
        return this.messagePrefix + '.' + FileSystem.MESSAGE_PREFIX + '.' + name;
    }
}
exports.default = FileSystem;
FileSystem.MESSAGE_PREFIX = 'file_system';
//# sourceMappingURL=FileSystem.js.map