---
title:  Offline Cache for storing data
author: Miroslav Behina
date:   22.11.2018
type:   js-api
tags:
- applet
- applet_api
- api
- js_api
description: "[Content JS API] This API provides you with approach similar to the HTML5's Local Storage, but implemented internally via native device API and completely device agnostic."
---

# Offline Cache for storing simple data

Use when you need to save some data into local memory. 
This API provides you with approach similar to the HTML5's Local Storage, but implemented internally via native device API and completely device agnostic. 
We do not recommend using a different storage type then Offline Cache or full featured [File System API](/api/js/content/js-file-system) that will provide you with more low-level methods.

## All methods
::: table-responsive
| Method | Description | Supported Since |
| ------ | -------- | ----- | :--------: |
| `loadContent()` | Loads content from internal storage | 1.0.3 |
| `saveContent()` | Saves content to internal storage | 1.0.3 |
| `listContents()` | List all content items saved previously to internal storage | 2.0.0 |
| `deleteContent()` | Delete content item previously saved to internal storage	 | 2.0.0 |
:::

::: alert alert--warning
Emulator has certain limitations while handling offline files. [Read more here](https://docs.signageos.io/hc/en-us/articles/4405238997138)
:::

## loadContent()
Method `loadContent()` loads content from internal storage.

### Parameters
::: table-responsive
|      Param     | Type   | Required | Description  |
| -------------- | ------ | :-------: | ----------- |
| `uid` | string | <div class="red">Yes</div> | Unique file identifier is used for later file retrieval, must contain a-z,A-Z,0-9 and . characters |             				  |
:::

### Javascript example
```javascript
await sos.offline.cache.saveContent('ApplicationSecret', '123SuperSecretHash');
```

## saveContent()
Method for saving content into internal storage. 

### Parameters
::: table-responsive
|      Param     | Type   | Required | Description  |
| -------------- | ------ | :-------: | ----------- |
| `uid` | string | <div class="red">Yes</div> | Unique file identifier is used for later file retrieval, must contain a-z,A-Z,0-9 and . characters |
| `content` | string | <div class="red">Yes</div> | Only string variables enabled. For JSON values use `JSON.Stringify()`               				  |
:::

### Javascript Example
```javascript
sos.offline.cache.saveContent('ApplicationSecret', '123SuperSecretHash')
.then(() => {
    //Content was successfully saved, retrieve it. 
    return sos.offline.cache.loadContent('ApplicationSecret');
})
.then((content) => {
    console.log('Loaded', content); // print 123SuperSecretHash
})
```

## listContents()
List all content items saved previously to internal storage

### Javascript example
```javascript
await sos.offline.cache.listContents();
```

## deleteContent()
Delete content item previously saved to internal storage.

### Parameters
::: table-responsive
|      Param     | Type   | Required | Description  |
| -------------- | ------ | :-------: | ----------- |
| `uid` | string | <div class="red">Yes</div> | Unique file identifier is used for later file retrieval, must contain a-z,A-Z,0-9 and . characters |             				  |
:::

### Javascript example
```javascript
//Store
sos.offline.cache.saveContent('ApplicationSecret', '123SuperSecretHash')
.then(() => {
    //Content was successfully saved, retrieve it. 
    return sos.offline.cache.loadContent('ApplicationSecret');
})
.then((content) => {
    console.log('Loaded', content); // print 123SuperSecretHash
 
    // Let's delete the content now
    return sos.offline.cache.deleteContent('ApplicationSecret')
})
.then(() => {
    console.log("Deleted");
})
.catch((error) => { console.error(error); });
```

## Usage with Typescript
You can also use all these methods with [signageOS TypeScript](https://docs.signageos.io/hc/en-us/articles/4405069154962#signageos-typescript).

```typescript
loadContent(uid: string): Promise<string>;
saveContent(uid: string, content: string): Promise<void>;

listContents(): Promise<string[]>;
deleteContent(uid: string): Promise<void>;
```

## Errors
Although we are doing our best, following errors may occur when working with the offline cache.

::: table-responsive
| Code | Type | Message |
| ---- | ---- | ------- |
| 41001 | AppletNativeCacheError | Already saving the file with UID: ```uid``` |
| 51001 | InternalNativeCacheError | Couldn't not read the files from the offline cache. |
| 51001 | InternalNativeCacheError | Couldn\'t load the file from offline cache. |
| 51001 | InternalNativeCacheError | Couldn\'t load the file before deleting it. |
| 51002 | InternalNativeCacheError | Couldn\'t save the file to the offline cache. |
| 51003 | InternalNativeCacheError | File wasn\'t deleted correctly. |
| 40901 | AppletOfflineCacheError | Uid contains invalid characters, allowed: $$ALLOWED_CHARS$$, got $$ACTUAL_UID$$ |
:::
