---
title:  Offline Cache for media files
author: Miroslav Behina
date:   22.11.2018
type:   js-api
tags:
- applet
- applet_api
- api
- js_api
description: "[Content JS API] File API allows files (video files, media files etc.) to be stored for offline usage."
---

# Offline Cache for media files (File API)
File API allows files (video files, media files etc.) to be stored for offline usage.

## All methods

::: table-responsive
| Methods | Description | Supported since |
| ------ | -------- | ----- | :-------------: |
| `listFiles()` | Return list of files of existing directory | 2.0.0 |
| `saveFile()` | Download file to selected directory | 1.0.3 |
| `loadFile()` | Load file from cache | 1.0.3 |
| `loadOrSaveFile()` | Download and load file from directory | 1.0.9 |
| `deleteFile()` | Delete file from directory | 2.0.0  |
| `getChecksumFile()` | Get checksum of a file | 2.0.0  |
| `validateChecksumFile()` | Validate that file's checksum matches expected value | 2.0.0  |
| `decompressFile()` | Decompress a compressed file or archive | 2.1.0  |
:::

::: alert alert--warning
Emulator has certain limitations while handling offline files. [Read more here](https://docs.signageos.io/hc/en-us/articles/4405238997138)
:::

## listFiles() 
This method is used to get a list of uids of all files, saved by the current applet.

### Javascript example
```javascript
// List uids of all files
await sos.offline.cache.listFiles().then((fileUids) => {
    console.log("Loaded", fileUids); 
}).catch((error) => { console.error(error); });
```
### Returns
```javascript
["video-3", "video-2", "video-1", "image-9", "image-8", "image-7", "image-6", "image-5", "image-4", "image-3", "image-2", "image-13", "image-12", "image-11", "image-10", "image-1"]
```


## saveFile()
Method `saveFile()` is used to save files from remote a destination into the device internal memory.

### Parameters

::: table-responsive
|      Param       | Type                  | Required | Description                  					|
| ---------------- | --------------------- | :------: |---------------------------------------------- |
| `uid`              | string                | <div class="red">Yes</div> | Unique file identifier is used for later file retrieval, must contain a-z,A-Z,0-9 and . characters, slash / is supported since Front-display version 6.0.0 |
| `uri`              | string                | <div class="red">Yes</div> | URL address to retrieve the file				|
| `headers`         | object                | <div class="yellow">No</div> | Key, value pairs of HTTP headers to send along with the request. Used when the target file is protected by a password or if any other specific headers are needed to access it.                                   			|
:::

::: alert alert--warning
`headers` has to be a JSON object. If you are passing the value, make sure you use `JSON.parse()`.
:::

::: alert alert--warning
`uid` should have the same file extension (e.g.: mp4, svg, jpg) as the original file.
:::

### Javascript example
```javascript
// Example saving files into internal memory, unique id could be salted md5 hash, uri directs to our CDN
await sos.offline.cache.saveFile('9d66725ba2105f1833731ade5b7f334e.mp4', 'https://cdn.my-cms.com/files/video.mp4', { "Authorization": "Basic Zm9vOmJhcg==" })
.then(() => { console.log('Saved'); })
.catch((error) => { console.error(error); });
```

::: alert alert--info
Local device file path differs from device to device. It can point to file:// or http://localhost etc.
:::

## loadFile()
Method `loadFile()` is used for individual file retrieval from internal memory.

### Parameters

| Param | Type | Required | Description |
| --------- | ------------ | :---------: | --------------- |
| `uid` | string | <div class="red">Yes</div> | Unique file identifier is used for later file retrieval, must contain a-z,A-Z,0-9 and . characters |

::: alert alert--warning
`uid` should have the same file extension (e.g.: mp4, svg, jpg) as the original file.
:::

### Javascript Example
```javascript
await sos.offline.cache.loadFile('9d66725ba2105f1833731ade5b7f334e.mp4')
.then((file) => { console.log('Loaded', file); })
.catch((error) => { console.error(error); });
 
// Logs into console
{
  filePath: '/real/device/path/to/file/9d66725ba2105f1833731ade5b7f334e.mp4',
}
```

::: alert alert--info
Local device file path differs from device to device. It can point to file:// or http://localhost etc.
:::

### Returns
* SUCCESS: File object
* FAIL: No file of such uid is available, it throws error into promise

## loadOrSaveFile()
Method `loadOrSaveFile()` is used for individual file retrieval & save in case when file is not saved in local storage yet.
To get file from internal memory & save it when not yet exists we prepared `loadOrSaveFile()` method:

::: alert alert--info
* The file URI has to return the file. If your URI leads to a 303 redirect (e.g. from http to http**s**), the API will not work.
* Emulator has certain limitations while handling offline files. [Read more here](https://docs.signageos.io/hc/en-us/articles/4405238997138) 
:::

### Parameters

|      Param      | Type           | Required | Description                  					|
| --------------- | -------------- | :-------: |---------------------------------------------- |
| `uid`             | string         | <div class="red">Yes</div> | Unique file identifier is used for later file retrieval, must contain a-z,A-Z,0-9 and . characters |
| `uri`             | string         | <div class="red">Yes</div> | URL address where to get the file             |
| `headers`        | object         | <div class="yellow">No</div> | Key, value pairs of HTTP headers to send along with the request. Used when the target file is protected by a password or if any other specific headers are needed to access it.                                   			|

::: alert alert--warning
`uid` should have the same file extension (e.g.: mp4, svg, jpg) as the original file.
:::

### Javascript Example
```javascript
await sos.offline.cache.loadOrSaveFile('9d66725ba2105f1833731ade5b7f334e.mp4', 'https://cdn.my-cms.com/files/video.mp4', { "Authorization": "Basic Zm9vOmJhcg==" })
.then((file) => { console.log('Loaded or Saved', file); })
.catch((error) => { console.error(error); });
 
// Logs into console
{
  filePath: '/real/device/path/to/file/9d66725ba2105f1833731ade5b7f334e.mp4',
}
```

::: alert alert--info
Local device file path differs from device to device. It can point to file:// or http://localhost etc.
:::

### Returns
* SUCCESS: File object
* FAIL: No file of such uid is available, it will download it to local storage & then return saved file.

<div class="row d-flex align-content-stretch force-padding-20">
    <div class="col-12 d-flex">
        <a class="wide-box wide-box--white d-flex align-content-stretch widebox-kb-color" target="_blank" href="https://github.com/signageos/applet-examples/blob/master/examples/content-js-api/offline-files">
            <div>
                <div class="wide-box__image d-flex align-items-center">
                    <i class="fab fa-github svg-black"></i>
                </div>
            <div>
                <h3 class="wide-box__title wide-box__title__small pl-85">Example of managing offline files</h3>
                <div class="wide-box__description">Working example on our Github</div>
            </div>
            </div>
        </a>
    </div>
</div>

<div class="row d-flex align-content-stretch force-padding-10">
    <div class="col-12 d-flex">
        <a class="wide-box wide-box--white d-flex align-content-stretch widebox-kb-color" target="_blank" href="https://github.com/signageos/applet-examples/blob/master/examples/content-js-api/video-loop-offline">
            <div>
                <div class="wide-box__image d-flex align-items-center">
                    <i class="fab fa-github svg-black"></i>
                </div>
            <div>
                <h3 class="wide-box__title wide-box__title__small pl-85">Example of playing video in loop</h3>
                <div class="wide-box__description">Working example on our Github</div>
            </div>
            </div>
        </a>
    </div>
</div>

## deleteFile()
Method `deleteFile()` is used for deleting previously saved file from internal memory.

### Parameters

|      Param     | Type         | Required  | Description                  					|
| -------------- | -------------| :-------: |-------------------------------------------- |
| `uid`            | string       | <div class="red">Yes</div> | Unique file identifier is used for later file retrieval, must contain a-z,A-Z,0-9 and . characters |

### Javascript example
```javascript
await sos.offline.cache.deleteFile('9d66725ba2105f1833731ade5b7f334e.mp4')
.then(() => { console.log('Deleted'); })
.catch((error) => { console.error(error); });
```

## getChecksumFile()
Method `getChecksumFile()` is used for getting the checksum of file from internal memory.

### Parameters:
|      Param     | Type            | Required  | Description                  					|
| -------------- | ----------------| :-------: |-------------------------------------------- |
| `uid`            | string          | <div class="red">Yes</div> |Unique file identifier is used for later file retrieval, must contain a-z,A-Z,0-9 and . characters |
| `hashType`       | string          | <div class="red">Yes</div> |Type of checksum has. Currently supported hashType is just md5 |

### Javascript example
```javascript
await sos.offline.cache.getChecksumFile('9d66725ba2105f1833731ade5b7f334e.mp4', 'md5')
.then((checksum) => { console.log('MD5 checksum is ' + checksum); })
.catch((error) => { console.error(error); });
```

<div class="row d-flex align-content-stretch force-padding-20">
    <div class="col-12 d-flex">
        <a class="wide-box wide-box--white d-flex align-content-stretch widebox-kb-color" target="_blank" href="https://github.com/signageos/applet-examples/blob/master/examples/content-js-api/md5-checksum">
            <div>
                <div class="wide-box__image d-flex align-items-center">
                    <i class="fab fa-github svg-black"></i>
                </div>
            <div>
                <h3 class="wide-box__title wide-box__title__small pl-85">Example of checking checksum of file</h3>
                <div class="wide-box__description">Working example on our Github</div>
            </div>
            </div>
        </a>
    </div>
</div>

## validateChecksumFile()
Method `validateChecksumFile()` is used for validating the checksum of the previously saved file in from internal memory.

### Parameters:

|      Param     | Type             | Required  | Description                  					|
| -------------- | -----------------| :-------: |-------------------------------------------- |
| `uid`            | string           | <div class="red">Yes</div> |Unique file identifier is used for later file retrieval, must contain a-z,A-Z,0-9 and . characters |
| `hash`           | string           | <div class="red">Yes</div> |Hexa decimal representation of file checksum to be validated against |
| `hashType`       | string           | <div class="red">Yes</div> |Type of checksum hash. Currently supported hashType is just md5 |

### Javascript example
```javascript
// validate file checksum against internal memory
await sos.offline.cache.validateChecksumFile(uid, '23dc936691f46d2bbef631f18a02f94f', 'md5')
.then((valid) => { console.log('MD5 checksum is ' + (valid ? 'valid' : 'not valid')); })
.catch((error) => { console.error(error); });
```

## decompressFile() (ZIP)
Method `decompressFile()` is used to decompress (extract ZIP) file previously saved to internal memory.

### Parameters

|      Param      | Type         | Required  | Description                  					                     |
| --------------- | ------------ | :-------: |----------------------------------------------------------------- |
| `zipUid`          | string       | <div class="red">Yes</div> | Unique file identifier of a previously downloaded ZIP file          |
| `destinationUid`  | string       | <div class="red">Yes</div> | Unique directory identifier (prefix of file) to extract ZIP file to |
| `method`          | string       | <div class="red">Yes</div> | Method of compression algorithm, currently only zip is supported    |

### Javascript example
```javascript
// validate file checksum against internal memory
await sos.offline.cache.decompressFile(zipUid, destinationDirectoryUid, 'zip')
    .then(() => { console.log('ZIP file extracted'); })
    .catch((error) => { console.error(error); });
```

<div class="row d-flex align-content-stretch force-padding-20">
    <div class="col-12 d-flex">
        <a class="wide-box wide-box--white d-flex align-content-stretch widebox-kb-color" target="_blank" href="https://github.com/signageos/applet-examples/blob/master/examples/content-js-api/offline-zip-decompress">
            <div>
                <div class="wide-box__image d-flex align-items-center">
                    <i class="fab fa-github svg-black"></i>
                </div>
            <div>
                <h3 class="wide-box__title wide-box__title__small pl-85">Example of decompressing ZIP files</h3>
                <div class="wide-box__description">Working example on our Github</div>
            </div>
            </div>
        </a>
    </div>
</div>

## More examples

* [Managing files](https://github.com/signageos/applet-examples/blob/master/examples/content-js-api/file-system)
* [Sample applet from Box](https://github.com/signageos/applet-examples/blob/master/index.html)
* All other examples are available in Knowledge Base under [Applet Examples](https://github.com/signageos/applet-examples/)

## Errors
Although we are doing our best, following errors may occur when working with the offline cache.

| Code | Type | Message |
| ---- | ---- | ------- |
| 40105 | AppletResourceError | Already loading file: `$$FILE_NAME$$` |
|^^|^^| Please, check your code for multiple occurrences of addFile/s methods. |
| 40106 | AppletResourceError | Already existing file: `$$FILE_NAME$$` |
|^^|^^| Please, check your code for occurrences of addFile/s methods on lines before this error. |
| 49902 | FileNotFoundError | File was not found `$$FILE_ID$$` |
| 40901 | AppletOfflineCacheError | Uid contains invalid characters, allowed: `$$ALLOWED_CHARS$$`, got `$$ACTUAL_UID$$` |
|^^|^^| Please, check if the used URL is correct. |
| 40902 | AppletOfflineCacheError | Invalid headers `$$HEADERS$$` |
| 50901 | InternalOfflineCacheError | Couldn't not read the files from the offline cache. |
| 50902 | InternalOfflineCacheError | File wasn't saved correctly. |
| 50903 | InternalOfflineCacheError | Reading the file from the offline cache failed. |
