export interface IEventPayload {
    type: string;
}
/**
 * Event is a representation of a single event that happened in the system.
 * It's a wrapper interface for a generic event payload.
 * It contains all the necessary metadata for the event.
 * This event is mostly used for event sourcing.
 */
interface IEvent<TPayload extends IEventPayload> {
    /**
     * Unique identifier of the event.
     * In MongoDB, it is stored as `_id` and is automatically generated.
     */
    id: string;
    /**
     * Unique identifier of the command that caused the event.
     * You can find it in {@link import("../CQRS/ICommand").id}.
     */
    commandId: string;
    /**
     * Type of the event.
     * It should be always the same as {@link IEventPayload['type']}.
     */
    type: string;
    /**
     * Identifier of the source of the event.
     * It's usually the same as the {@link import("../CQRS/ICommand").sourceUid}.
     * It's usually used for logging and debugging.
     * It can be used in some cases to identify the source of the event
     * and to check if the event was dispatched by the same source as the command.
     */
    sourceUid: string;
    /**
     * Date when the original command was dispatched.
     * It's the same as {@link import("../CQRS/ICommand").receivedAt}.
     * It can differ when the event is not produced by the command.
     */
    dispatchedAt: Date;
    /**
     * Date when the event was received (written) to event store.
     * This date is the most important date that should be used
     * as a replacement for the `Date.now()` function.
     * Since event sourcing is used as a source of truth,
     * for consistency, all dates should be based on this date.
     */
    receivedAt: Date;
    /**
     * All events are ordered by the `receivedAt` date.
     * However, in some cases, there are multiple events emitted with the same `receivedAt` date.
     * It's the case where a single command emits multiple events.
     * In this case, the `secondaryIndex` is used to order the events.
     * So it's the important part of the ordering of the events.
     * See also {@link import("../CQRS/commandQueue").ITimestamp}.
     */
    secondaryIndex: number;
    /**
     * Lock keys are used to identify affected domains of the event.
     * It's used for specifying the dependencies between the events.
     * Only those events within the same lock key are dependent on each other.
     * So the order of the events is guaranteed only within the same lock key.
     * Events with different lock keys can be processed in parallel.
     */
    lockKeys: string[];
    /**
     * The generic payload of the event.
     * The only required field is the `type` field.
     * The `type` field should always be the same as the `type` field of the event.
     */
    payload: TPayload;
}
export default IEvent;
