import { IAMQPConnection } from '../amqpConnectionFactory';
import ICommand, { ICommandPayload } from './ICommand';
import ICommandError from './ICommandError';
import IQueueOptions from '../IQueueOptions';
export declare const QUEUE_NAME = "commands";
export declare const AUTO_SNAPSHOTS_QUEUE_NAME = "auto_snapshots";
export type BindOptions = Pick<IQueueOptions, 'prefetchCount'>;
export interface IMessageOptions {
    priority?: number;
    /** Commands are persistent by default but it can be changed for a single command */
    persistent?: boolean;
}
export type IResponse<TPayload extends ICommandPayload, TError extends ICommandError> = IProcessSucceed<TPayload> | IProcessFailed<TPayload, TError> | IError;
export interface ITimestamp {
    /**
     * The date represented as timestamp in milliseconds, when the command was processed and all emitted events was created.
     * More events emitted by the same command will have the same receivedAt timestamp. To determine order of event, see secondaryIndex.
     * It uses javascript date in ISO8601 format (with Z - zero hour offset always). The only important information is UNIX timestamp in ms.
     * Timezone offset is overlooked every time because of bad design of JS Date object.
     */
    receivedAt: Date;
    /**
     * This value differentiate more events created by same command (they have same receivedAt timestamp).
     * Using this index, you can determine the right order of events in the event store.
     * Index beginning from 0 and is increased by one for every following event.
     */
    secondaryIndex: number;
}
export interface IProcessSucceed<TPayload extends ICommandPayload> {
    status: 'process_succeed';
    dispatchedAt: Date;
    command: ICommand<TPayload>;
    /**
     * List of lock keys which are used to lock the event store for the command processing.
     * The lock keys are used to synchronize the command processing between multiple consumers.
     */
    lockKeys: string[];
    /**
     * Timestamp of the last event created by successfully processed command.
     * This timestamp can be used to detect synchronization status of consumers.
     * When timestamp is null, the event store was not changed. No new event was emitted, so consumers are up-to-date.
     */
    timestamp: ITimestamp | null;
}
export interface IProcessFailed<TPayload extends ICommandPayload, TError extends ICommandError> {
    status: 'process_failed';
    dispatchedAt: Date;
    command: ICommand<TPayload>;
    message: string;
    error: TError;
}
export interface IError {
    status: 'error';
    dispatchedAt: Date;
    error: any;
}
export declare function enqueue<TPayload extends ICommandPayload>(amqpConnection: IAMQPConnection, command: ICommand<TPayload>, messageOptions?: IMessageOptions): Promise<void>;
export declare function process<TPayload extends ICommandPayload, TCommandError extends ICommandError = ICommandError>(amqpConnection: IAMQPConnection, command: ICommand<TPayload>, messageOptions?: IMessageOptions): Promise<IResponse<TPayload, TCommandError>>;
export declare function bindAll<TPayload extends ICommandPayload>(amqpConnection: IAMQPConnection, options: BindOptions, onCommand: (command: ICommand<TPayload>) => Promise<IResponse<TPayload, ICommandError> | undefined>): Promise<() => Promise<import("../QueueSubscriber").ICloseChannel>>;
export declare function bindAutoSnapshots<TPayload extends ICommandPayload>(amqpConnection: IAMQPConnection, options: BindOptions, onCommand: (command: ICommand<TPayload>) => Promise<void>): Promise<() => Promise<import("../QueueSubscriber").ICloseChannel>>;
export declare function fetchNext<TPayload extends ICommandPayload>(amqpConnection: IAMQPConnection): Promise<ICommand<TPayload> | null>;
export declare function fetchNextAutoSnapshot<TPayload extends ICommandPayload>(amqpConnection: IAMQPConnection): Promise<ICommand<TPayload> | null>;
export declare function purgeAll(amqpConnection: IAMQPConnection): Promise<void>;
export declare function deleteAll(amqpConnection: IAMQPConnection): Promise<void>;
export declare function prepareAll(amqpConnection: IAMQPConnection): Promise<void>;
