"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createService = createService;
const httpServerFactory_1 = require("./HTTP/httpServerFactory");
const MetricsClient_1 = require("../Metrics/MetricsClient");
const connections_1 = require("./metrics/connections");
const Service_1 = require("./Service");
/**
 * Create instance of a service
 *
 * Service implements all functionality that is common for all services
 * This functionality is the same in every service.
 *
 * We had a problem of writing it from scratch each time for every service.
 * Sometimes there was a drift, where some new ideas were implemented in one or more services
 * but not in all services.
 *
 * This single common implementation aims to prevent that and provide a common implementation
 * that all services can use as a library.
 *
 * @param config - It's recommended to use the direct output of `parseServiceConfig()`. The format is deliberately the same.
 * @param connections - Database connections
 * @param isReady - Custom readiness check for Kubernetes
 * @param isAlive - Custom aliveness check for Kubernetes
 * @param accessLogCallback - Custom callback to modify the access log before it is logged
 *
 * -------
 *
 * Adding application-specific HTTP routes:
 *
 * This factory returns express app, that can be used to add more routes.
 *
 * For example:
 *
 *   ```typescript
 *   const service = createService({ ... });
 *   service.httpServer.express.get('/my-route', (req, res) => {
 *     res.send('Hello World!');
 *   });
 *   ```
 *
 * -------
 *
 * Adding application-specific metrics:
 *
 * By default, the service serves some basic Node.js metrics.
 * This factory returns a metrics client, that can be used to add more metrics.
 *
 * For example:
 *
 *  ```typescript
 *  const service = createService({ ... });
 *  const client = service.metricsClient.getClient();
 *  const myCounter = new client.Counter({
 *    name: 'my_counter',
 *    help: 'This is my counter',
 *  });
 *
 *  // now you can independently increment the counter in your code
 *  // and the value will automatically be exposed when serving metrics
 *  ```
 *
 *  -------
 *
 *  Custom readiness and aliveness checks:
 *
 * By default, the service serves `/ready` and `/alive` routes, that are used by Kubernetes readiness/aliveness probes.
 * You can provide custom callbacks if the default checks are not enough:
 *
 *  ```typescript
 *  const service = createService({
 *    // ...
 *
 *    isReady: async () => {
 * 			// your custom readiness check
 * 			// either return true/false if service is/isn't ready to receive traffic
 * 			// or return an object { ready: boolean, response: any } to provide a custom response
 * 			return true;
 * 		},
 *
 *    isAlive: async () => {
 * 			// your custom aliveness check
 * 			// return true/false if service is/isn't alive
 * 			return true;
 * 		},
 *  });
 *  ```
 */
function createService({ config, connections, isReady, isAlive, accessLogCallback, noRouteIndex, setDefaultPenTestHeaders, }) {
    const metricsClient = new MetricsClient_1.MetricsClient();
    const serviceInfo = {
        name: config.app.name,
        version: config.app.version,
        uid: process.uid,
    };
    const httpServer = (0, httpServerFactory_1.createHttpServer)({
        service: serviceInfo,
        port: config.app.port,
        monitoringPort: config.app.monitoringPort,
        metricsClient,
        connections,
        isReady,
        isAlive,
        accessLogCallback,
        noRouteIndex,
        setDefaultPenTestHeaders,
    });
    if (connections) {
        (0, connections_1.initDbConnectionsMetrics)(connections, metricsClient);
    }
    return new Service_1.Service({
        httpServer,
        metricsClient,
    });
}
//# sourceMappingURL=serviceFactory.js.map