"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDBConnection = void 0;
const mongoose_1 = __importDefault(require("mongoose"));
const debug_1 = __importDefault(require("debug"));
const proprietaryCluster_1 = require("./proprietaryCluster");
const logging_1 = require("../Logging/logging");
mongoose_1.default.Promise = Promise;
const debug = (0, debug_1.default)('@signageos/lib:MongoDB:mongoDBConnectionFactory');
const logger = logging_1.logger.createNamespace('MongoDBConnection');
const defaultOptions = {
    autoIndex: false,
    connectTimeoutMS: 10 * 60 * 1e3, // 10m
    socketTimeoutMS: 3 * 60 * 60 * 1e3, // 3h
    monitorCommands: true,
};
class MongoDBConnection {
    constructor(mongoDbDsn, options = {}) {
        this.mongoDbDsn = mongoDbDsn;
        this.connection = null;
        this.metrics = null;
        this.options = Object.assign(Object.assign({}, defaultOptions), options);
    }
    connect() {
        return __awaiter(this, void 0, void 0, function* () {
            debug('connect');
            const updatedMongDbDsn = yield (0, proprietaryCluster_1.updateDsnIfProprietaryCluster)(this.mongoDbDsn);
            const connection = mongoose_1.default.createConnection(updatedMongDbDsn, this.options);
            this.logMongoClient(connection.getClient());
            if (this.metrics) {
                this.monitorConnection(connection, this.metrics.exporter, this.metrics.connectionName);
            }
            yield connection.asPromise();
            debug('connected');
            this.connection = connection;
        });
    }
    close() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.connection) {
                yield this.connection.close();
            }
        });
    }
    monitor(exporter, connectionName) {
        if (this.connection) {
            debug(`monitor connection ${connectionName} now`);
            this.monitorConnection(this.connection, exporter, connectionName);
        }
        else {
            debug(`monitor connection ${connectionName}, once it connects`);
            // if connection is not yet established, store the exporter and name for later use
            this.metrics = {
                exporter,
                connectionName,
            };
        }
    }
    /**
     * @see https://www.mongodb.com/docs/drivers/node/current/fundamentals/monitoring/command-monitoring/#command-monitoring
     */
    monitorConnection(connection, exporter, connectionName) {
        debug(`monitor connection ${connectionName}`);
        connection.on('connected', () => logger.info('connected', { connectionName }));
        connection.on('disconnected', () => logger.error('disconnected', { connectionName }));
        connection.on('error', (error) => logger.error('error', { connectionName, error }));
        connection.on('close', () => logger.error('closed', { connectionName }));
        const mongoClient = connection.getClient();
        exporter.monitorMongoClient(mongoClient, connectionName);
        const eventNames = ['commandStarted', 'commandSucceeded', 'commandFailed'];
        for (const eventName of eventNames) {
            mongoClient.addListener(eventName, (event) => {
                debug(eventName, 
                // https://github.com/GoogleChromeLabs/jsbi/issues/30
                JSON.stringify(event, (_key, value) => (typeof value === 'bigint' ? value.toString() : value)));
            });
        }
    }
    logConnectionEvent(event, level = 'debug') {
        var _a;
        // name is not defined in interface, but it is there in most cases
        const logMessage = `MongoDB event${event.name ? `: '${event.name}'` : ''}`;
        const connectionName = (_a = this.metrics) === null || _a === void 0 ? void 0 : _a.connectionName;
        if (level === 'debug') {
            debug(logMessage, { event, connectionName });
            return;
        }
        logger[level](logMessage, { event, connectionName });
    }
    logMongoClient(mongoClient) {
        mongoClient.on('connectionPoolCreated', (event) => this.logConnectionEvent(event));
        mongoClient.on('connectionPoolClosed', (event) => this.logConnectionEvent(event, 'error'));
        mongoClient.on('connectionCreated', (event) => this.logConnectionEvent(event));
        mongoClient.on('connectionClosed', (event) => this.logConnectionEvent(event, 'error'));
        mongoClient.on('connectionCheckOutStarted', (event) => this.logConnectionEvent(event));
        mongoClient.on('connectionCheckedOut', (event) => this.logConnectionEvent(event));
        mongoClient.on('connectionCheckOutFailed', (event) => this.logConnectionEvent(event, 'error'));
        mongoClient.on('connectionCheckedIn', (event) => this.logConnectionEvent(event));
        mongoClient.on('commandSucceeded', (event) => this.logConnectionEvent(event));
        mongoClient.on('commandFailed', (event) => this.logConnectionEvent(event, 'debug'));
        mongoClient.on('serverClosed', (event) => this.logConnectionEvent(event, 'error'));
        mongoClient.on('serverOpening', (event) => this.logConnectionEvent(event));
        mongoClient.on('serverHeartbeatFailed', (event) => this.logConnectionEvent(event, 'error'));
        mongoClient.on('topologyDescriptionChanged', (event) => this.logConnectionEvent(event, 'error'));
    }
}
exports.MongoDBConnection = MongoDBConnection;
//# sourceMappingURL=MongoDBConnection.js.map