/**
 * This type will return all keys of object that are matching the specified type as an value.
 *
 * E.g.:
 * ```ts
 * type MyObject = {
 * 	a: string;
 * 	b: number;
 * };
 * type MyKeys = KeysMatching<MyObject, string>; // will be "a"
 * ```
 */
export type KeysMatching<T, V> = {
    [K in keyof T]-?: T[K] extends V ? K : never;
}[keyof T];
export type KeysMatchingUndefined<T> = {
    [K in keyof T]-?: undefined extends T[K] ? K : never;
}[keyof T];
/**
 * This type will return the same type input one. However, all keys that have their values possibly undefined
 * will be marked as optional as well.
 *
 * E.g.:
 * ```ts
 * type A = {
 * 	a: string;
 * 	b: number | undefined;
 * };
 * type B = OptionalUndefinedKeys<A>; // will be { a: string; b?: number | undefined; }
 * ```
 */
export type PartialFromUndefined<T extends object> = {
    [Key in KeysMatchingUndefined<T>]+?: T[Key];
} & {
    [Key in Exclude<keyof T, KeysMatchingUndefined<T>>]: T[Key];
};
/**
 * It does the mathematical set difference of two types recursively.
 * So let's imagine type A and type B. It does "C = A - B".
 * Because of limitation of TS, it has to get the placeholder value type as 3rd argument.
 * So the type B has to have all values (except the recursive objects) as the specified placeholder value.
 *
 * E.g.:
 * ```ts
 * type A = {
 * 	a: string;
 * 	b: {
 * 		c: number;
 * 		d: string;
 * 	};
 * };
 * type B = {
 * 	b: {
 * 		c: Function;
 * 	};
 * };
 * type C = ObjectRecursiveDifference<A, B, Function>; // will be { a: string; b: { d: string; }; }
 * ```
 */
export type ObjectRecursiveDifference<ObjectA, ObjectB, PlaceholderValueOfObjectB> = PartialFromUndefined<{
    [Key in Exclude<keyof ObjectA, KeysMatching<ObjectB, PlaceholderValueOfObjectB>>]: ObjectA[Key] extends object ? Key extends keyof ObjectB ? ObjectRecursiveDifference<ObjectA[Key], ObjectB[Key], PlaceholderValueOfObjectB> : ObjectA[Key] : ObjectA[Key];
}>;
/**
 * It does the mathematical set intersection of two types recursively.
 * So let's imagine type A and type B. It does "C = A ∩ B".
 * Because of limitation of TS, it has to get the placeholder value type as 3rd argument.
 * So the type B has to have all values (except the recursive objects) as the specified placeholder value.
 *
 * E.g.:
 * ```ts
 * type A = {
 * 	a: string;
 * 	b: {
 * 		c: number;
 * 		d: string;
 * 	};
 * };
 * type B = {
 * 	b: {
 * 		c: Function;
 * 	};
 * };
 * type C = ObjectRecursiveIntersection<A, B, Function>; // will be { b: { c: number; }; }
 * ```
 */
export type ObjectRecursiveIntersection<ObjectA, ObjectB, PlaceholderValueOfObjectB> = PartialFromUndefined<{
    [Key in Extract<keyof ObjectA, KeysMatching<ObjectB, PlaceholderValueOfObjectB>>]: ObjectA[Key] extends object ? Key extends keyof ObjectB ? ObjectRecursiveDifference<ObjectA[Key], ObjectB[Key], PlaceholderValueOfObjectB> : ObjectA[Key] : ObjectA[Key];
}>;
