"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MacAddressSchema = exports.NameSchema = exports.UsernameSchema = exports.TimeSchema = exports.NumericIdSchema = exports.DomainSchema = exports.HexColorSchema = exports.LegacyUIDSchema = exports.UIDSchema = void 0;
const zod_1 = require("zod");
const isHexColor_1 = require("validator/lib/isHexColor");
const isFQDN_1 = require("validator/lib/isFQDN");
const uid_1 = require("@signageos/common-types/dist/Entity/uid");
/**
 * The new version of the UID generated using NanoID package.
 */
exports.UIDSchema = zod_1.z
    .string()
    .length(uid_1.UID_LENGTH)
    .regex(new RegExp(`^[${uid_1.UID_CHARACTERS}]+$`), {
    message: `The UID is not a valid NanoID with exactly ${uid_1.UID_LENGTH} characters containing only the characters "${uid_1.UID_CHARACTERS}"`,
});
/**
 * The old version of UID generated in our system. It has hexadecimal characters and 50 characters length.
 * It used to be generated using generateUniqueHash() function of the @signageos/lib package.
 * Use the generators.ts -> uid() function to generate new UIDs.
 */
exports.LegacyUIDSchema = zod_1.z
    .string()
    .length(50)
    .regex(/^[0-9a-f]+$/);
exports.HexColorSchema = zod_1.z.string().refine(isHexColor_1.default, 'not a valid hex color');
exports.DomainSchema = zod_1.z.string().refine(isFQDN_1.default, 'not a valid domain');
/**
 * The old numeric id value generated in our system.
 * Primarily was used because of postgresql database (i.e. accountId in account table).
 */
exports.NumericIdSchema = zod_1.z
    .union([zod_1.z.number(), zod_1.z.string().regex(/^\d+$/)])
    .transform((value) => (typeof value === 'string' ? parseInt(value) : value));
exports.TimeSchema = zod_1.z
    .string()
    .regex(/^([01]\d|2[0-3]):[0-5]\d(:[0-5]\d)?$/, 'Time must be in the format HH:mm:ss where :ss is optional');
exports.UsernameSchema = zod_1.z
    .string()
    .min(3)
    .max(50)
    .regex(/^[a-zA-Z0-9_\-\.\+]*$/, 'Username can only contain letters, numbers, and the following characters: _ - . +');
exports.NameSchema = zod_1.z
    .string()
    .max(100)
    .refine((value) => {
    if (value.match(/(\.\w+)+/)) {
        return false;
    }
    if (value.match(/:\/\//)) {
        return false;
    }
    return true;
}, 'Name cannot contain a dot following a word and special reserved letters');
exports.MacAddressSchema = zod_1.z
    .string()
    .refine((value) => {
    const macAddressRegex = /^([0-9A-Fa-f]{2}([:-]| )?){6}$/;
    return macAddressRegex.test(value);
}, 'Mac address must be in the format XX:XX:XX:XX:XX:XX where X is a hexadecimal character')
    .transform((value) => {
    return value
        .replace(/[:-\s]/g, '')
        .replace(/(.{2})/g, '$1:')
        .slice(0, -1)
        .toLowerCase();
});
//# sourceMappingURL=schemas.js.map