import { z } from 'zod';
import { ObjectRecursiveDifference, ObjectRecursiveIntersection } from './types';
import { BaseCommandSchema } from './commandSchemas';
/** Generator function of property for commands. It can return any value. E.g.: uid (available in generators.ts) */
export type PropGeneratorWithPayload<TCommand> = (payload: TCommand) => unknown;
export type PropGenerator = () => unknown;
/**
 * Key-value list of generators used to be automatically generated to the command under the specified key. It can be recursive.
 *
 * E.g.:
 * ```ts
 * const generators: PropGenerators = {
 * 	uid,
 *  nested: {
 * 		uid,
 * 	},
 * };
 * ```
 */
type PropGenerators<TCommand> = {
    [K in keyof TCommand]?: PropGeneratorWithPayload<TCommand> | PropGenerator | PropGenerators<TCommand[K]>;
};
/**
 * Represents the data of a specific command excluding the type property.
 * The mandatority of properties are kept from the schema.
 * Except the properties that are generated automatically using the generators. Those are optional.
 */
type CommandData<TSchema extends BaseCommandSchema<Type>, Type extends string, TGenerators extends PropGenerators<z.infer<TSchema>>> = ObjectRecursiveDifference<z.input<TSchema>, TGenerators & {
    type: PropGeneratorWithPayload<TSchema> | PropGenerator;
}, Function> & Partial<ObjectRecursiveIntersection<z.infer<TSchema>, TGenerators, Function>>;
/**
 * Factory function for generating commands based on given schema and generators.
 * Represents the result of the createCommandFactory function.
 * It is used to generate commands of the specified type.
 */
type CommandFactory<TSchema extends BaseCommandSchema<Type>, Type extends string, TGenerators extends PropGenerators<z.infer<TSchema>>> = (data: CommandData<TSchema, Type, TGenerators>) => z.infer<TSchema>;
export interface ICommandFactoryOptions {
    /**
     * By default, all command schemas are registered in the global command schema registry when the command factory is created.
     * Using this option, you can skip global registration of the command schema.
     * It is useful for example for union schemas, where you want to register only the union schema, not the individual schemas.
     * Use this option only if you know what you are doing with caution.
     */
    noRegisterCommandSchema?: boolean;
}
/**
 * This is abstraction of the command factory. It should be used to create command factory for each command type.
 *
 * All command types should create it's own command factory using this function.
 * It has to be created from existing zod schema. And optionally from property generators.
 * Optional property generators are used to generate some specific properties automatically.
 * For example it can generate random UIDs for the command (it can be nested recursively in the payload).
 *
 * E.g.:
 * ```ts
 * export const CreateOrganization = "Organization.CreateOrganization";
 * export const CreateOrganizationSchema = z.strictObject({
 * 	type: z.literal(CreateOrganization),
 * 	uid: UIDSchema,
 * 	// ...
 * });
 * export type CreateOrganization = z.infer<typeof CreateOrganizationSchema>;
 * export const CreateOrganizationFactory = createCommandFactory(CreateOrganizationSchema, { uid });
 * ```
 *
 * @param schema Zod schema of the command.
 * @param generators See the PropGenerators type for more info.
 * @returns Factory function for generating commands based on given schema and generators.
 */
export declare function createCommandFactory<TSchema extends BaseCommandSchema<Type>, Type extends string, TGenerators extends PropGenerators<z.infer<TSchema>> = {}>(schema: TSchema, generators?: TGenerators, options?: ICommandFactoryOptions): CommandFactory<TSchema, Type, TGenerators>;
export {};
