"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EntitySchema = void 0;
exports.createCreateEntityCommandSchema = createCreateEntityCommandSchema;
exports.createCreateEntitiesCommandSchema = createCreateEntitiesCommandSchema;
exports.createUpdateEntityCommandSchema = createUpdateEntityCommandSchema;
exports.createDeleteEntityCommandSchema = createDeleteEntityCommandSchema;
const zod_1 = require("zod");
const ICommand_1 = require("../ICommand");
const schemas_1 = require("../schemas");
/**
 * Basic entity schema that all commands that work with an entity should extend
 */
exports.EntitySchema = zod_1.z.strictObject({
    uid: schemas_1.UIDSchema,
});
/**
 * Takes entity schema and generates a schema of a create command for that entity
 *
 * Here's how the create command schema is generated from the original entity schema:
 *   * the entire entity schema is kept as is
 *   * type literal equal to "type" arg is added
 *   * OriginatorAwareCommandSchema is added
 */
function createCreateEntityCommandSchema(type, schema) {
    return ICommand_1.OriginatorAwareCommandSchema.merge(schema).extend({
        type: zod_1.z.literal(type),
    });
}
/**
 *
 * @param type - type literal of a command
 * @param schema - schema to validate against
 * @returns - new schema with an array of entities
 */
function createCreateEntitiesCommandSchema(type, schema) {
    return ICommand_1.OriginatorAwareCommandSchema.merge(zod_1.z.object({ items: zod_1.z.array(schema) })).extend({
        type: zod_1.z.literal(type),
    });
}
/**
 * Takes entity schema and generates a schema of an update command for that entity
 *
 * Here's how the update command schema is generated from the original entity schema:
 *   * uid stays the same
 *   * type literal equal to "type" arg is added
 *   * OriginatorAwareCommandSchema is added
 *   * all other fields are changed to optional so that any subset of the entity's fields can be updated
 *   * all fields that are optional in the original entity schema are changed to optional or null
 *     where null means that the fields would be unset (it's optional so it should be possible to unset it)
 */
function createUpdateEntityCommandSchema(type, schema) {
    // There is a number of cases where some entities were created with a different uid format before uid format became strict.
    // We want to enforce creating new entities with the new uid format but we also want to allow updating entities with the old uid format.
    // So, in order to achieve that, update command will also accept LegacyUID format.
    // It's safe because update only affects entities that already exist while creating new entities will always use the new uid format.
    const uidSchema = schema.shape.uid.or(schemas_1.LegacyUIDSchema);
    let resultSchema = ICommand_1.OriginatorAwareCommandSchema.extend({
        type: zod_1.z.literal(type),
        uid: uidSchema,
    });
    for (const key of Object.keys(schema.shape)) {
        const shapePart = schema.shape[key];
        if (shapePart.isOptional()) {
            resultSchema = resultSchema.merge(zod_1.z.strictObject({
                [key]: shapePart.nullable(),
            }));
        }
        else if (key !== 'uid') {
            resultSchema = resultSchema.merge(zod_1.z.strictObject({
                [key]: shapePart.optional(),
            }));
        }
    }
    // Necessary type cast since the types here are very complicated.
    // The important part is that the return type is derived from the input schema
    // so the callers of this function will work with a correct type.
    // Functionality of this function will be verified by enough unit tests to compensate for the type mess.
    // TODO make the types work without ugly type casts
    return resultSchema;
}
/**
 * Takes entity schema and generates a schema of a delete command for that entity
 *
 * Here's how the create command schema is generated from the original entity schema:
 *   * only uid is taken from the schema since that's all that's necessary to reference the entity
 *   * type literal equal to "type" arg is added
 *   * OriginatorAwareCommandSchema is added
 */
function createDeleteEntityCommandSchema(type, schema) {
    return zod_1.z
        .strictObject({
        uid: schema.shape.uid,
    })
        .merge(ICommand_1.OriginatorAwareCommandSchema)
        .extend({
        type: zod_1.z.literal(type),
    });
}
//# sourceMappingURL=entityCommands.js.map