import { z } from 'zod';
import { SubscriptionType } from '@signageos/common-types/dist/Subscription/subscription';
import IBaseCommand, { IOriginatorAwareCommand } from '../ICommand';
import { PropGeneratorWithPayload } from '../commandFactory';
export declare const DeviceUidSchema: z.ZodString;
/**
 * Unique device identifier, generated by the device.
 *
 * Device generates its own device uid in a way that's deterministic and reproducible
 * so that even if the device is completely erased or factory reset,
 * it will generate the same deviceUid.
 * This is to ensure that the device always reports to us as the same device.
 *
 * Usually, deviceUid is derived from static values that don't change
 * and are unique for the device like serial number or ethernet mac address.
 */
type DeviceUid = z.infer<typeof DeviceUidSchema>;
export declare const DeviceIdentityHashSchema: z.ZodString;
/**
 * Unique device identifier, generated by our system.
 *
 * We generate identity hash to identify device within our system.
 * Unlike device uid, identity hash always has the same format and length
 * and is suitable to be used as the private key in most databases.
 *
 * Using identity hash is preferable to device uid in most cases when identifying devices.
 * There are few exceptions though.
 */
type DeviceIdentityHash = z.infer<typeof DeviceIdentityHashSchema>;
export interface IDevice {
    id: number;
    uid: DeviceUid;
    createdAt: Date;
    name: string;
    timezone: number;
    aliveAt: Date | null;
    connectionsCount: number | null;
}
export declare const SetDevicePropertySchema: z.ZodObject<{
    context: z.ZodOptional<z.ZodObject<{
        whiteLabelSettingsUid: z.ZodOptional<z.ZodUnion<[z.ZodString, z.ZodString]>>;
        auth0Enabled: z.ZodOptional<z.ZodBoolean>;
    }, "strict", z.ZodTypeAny, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }>>;
    originator: z.ZodObject<{
        accountId: z.ZodOptional<z.ZodNumber>;
        organizationUid: z.ZodOptional<z.ZodString>;
    }, "strict", z.ZodTypeAny, {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    }, {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    }>;
    deviceIdentityHash: z.ZodString;
}, "strict", z.ZodTypeAny, {
    originator: {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    };
    deviceIdentityHash: string;
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
}, {
    originator: {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    };
    deviceIdentityHash: string;
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
}>;
export type TSetDeviceProperty = z.infer<typeof SetDevicePropertySchema>;
/**
 * @deprecated use SetDevicePropertySchema (zod) instead without generics. As it is more strict and have run time validation
 */
export interface ISetDeviceProperty<T extends string> extends IOriginatorAwareCommand<T> {
    deviceIdentityHash: DeviceIdentityHash;
}
export declare const MarkSetDevicePropertyDoneSchema: z.ZodObject<{
    context: z.ZodOptional<z.ZodObject<{
        whiteLabelSettingsUid: z.ZodOptional<z.ZodUnion<[z.ZodString, z.ZodString]>>;
        auth0Enabled: z.ZodOptional<z.ZodBoolean>;
    }, "strict", z.ZodTypeAny, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }>>;
    uid: z.ZodString;
}, "strict", z.ZodTypeAny, {
    uid: string;
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
}, {
    uid: string;
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
}>;
/**
 * @deprecated use MarkSetDevicePropertyDoneSchema (zod) instead without generics. As it is more strict and have run time validation
 */
export interface IMarkSetDevicePropertyDone<T extends string> extends IBaseCommand<T> {
    uid: string;
}
export type StartEditDeviceNameType = 'Device.StartEditDeviceName';
export declare const StartEditDeviceName: StartEditDeviceNameType;
export interface StartEditDeviceName extends IBaseCommand<StartEditDeviceNameType> {
    deviceUid: DeviceUid;
}
export type UpdateDeviceNameType = 'Device.UpdateDeviceName';
export declare const UpdateDeviceName: UpdateDeviceNameType;
export interface UpdateDeviceName extends IOriginatorAwareCommand<UpdateDeviceNameType> {
    deviceUid: DeviceUid;
    name: string;
}
export type CreateDeviceType = 'Device.CreateDevice';
export declare const CreateDevice: CreateDeviceType;
export declare const CreateDeviceSchema: z.ZodObject<{
    context: z.ZodOptional<z.ZodObject<{
        whiteLabelSettingsUid: z.ZodOptional<z.ZodUnion<[z.ZodString, z.ZodString]>>;
        auth0Enabled: z.ZodOptional<z.ZodBoolean>;
    }, "strict", z.ZodTypeAny, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }>>;
    type: z.ZodLiteral<"Device.CreateDevice">;
    uid: z.ZodString;
    applicationType: z.ZodEnum<["sssp", "tizen", "webos", "android", "chrome", "brightsign", "linux", "windows", "default", "chromeos"]>;
    identityHash: z.ZodOptional<z.ZodString>;
    originator: z.ZodOptional<z.ZodObject<{
        accountId: z.ZodOptional<z.ZodNumber>;
        organizationUid: z.ZodOptional<z.ZodString>;
    }, "strict", z.ZodTypeAny, {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    }, {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    }>>;
}, "strict", z.ZodTypeAny, {
    type: "Device.CreateDevice";
    uid: string;
    applicationType: "default" | "sssp" | "tizen" | "webos" | "android" | "chrome" | "brightsign" | "linux" | "windows" | "chromeos";
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    identityHash?: string | undefined;
    originator?: {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    } | undefined;
}, {
    type: "Device.CreateDevice";
    uid: string;
    applicationType: "default" | "sssp" | "tizen" | "webos" | "android" | "chrome" | "brightsign" | "linux" | "windows" | "chromeos";
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    identityHash?: string | undefined;
    originator?: {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    } | undefined;
}>;
export type CreateDevice = z.infer<typeof CreateDeviceSchema>;
export declare const CreateDeviceFactory: (data: {
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    originator?: {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    } | undefined;
} & {
    uid: string;
    applicationType: "default" | "sssp" | "tizen" | "webos" | "android" | "chrome" | "brightsign" | "linux" | "windows" | "chromeos";
} & Partial<import("../types").ObjectRecursiveIntersection<{
    type: "Device.CreateDevice";
    uid: string;
    applicationType: "default" | "sssp" | "tizen" | "webos" | "android" | "chrome" | "brightsign" | "linux" | "windows" | "chromeos";
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    identityHash?: string | undefined;
    originator?: {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    } | undefined;
}, {
    identityHash: PropGeneratorWithPayload<{
        type: "Device.CreateDevice";
        uid: string;
        applicationType: "default" | "sssp" | "tizen" | "webos" | "android" | "chrome" | "brightsign" | "linux" | "windows" | "chromeos";
        context?: {
            whiteLabelSettingsUid?: string | undefined;
            auth0Enabled?: boolean | undefined;
        } | undefined;
        identityHash?: string | undefined;
        originator?: {
            accountId?: number | undefined;
            organizationUid?: string | undefined;
        } | undefined;
    }>;
}, Function>>) => z.TypeOf<z.ZodObject<{
    context: z.ZodOptional<z.ZodObject<{
        whiteLabelSettingsUid: z.ZodOptional<z.ZodUnion<[z.ZodString, z.ZodString]>>;
        auth0Enabled: z.ZodOptional<z.ZodBoolean>;
    }, "strict", z.ZodTypeAny, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }>>;
    type: z.ZodLiteral<"Device.CreateDevice">;
    uid: z.ZodString;
    applicationType: z.ZodEnum<["sssp", "tizen", "webos", "android", "chrome", "brightsign", "linux", "windows", "default", "chromeos"]>;
    identityHash: z.ZodOptional<z.ZodString>;
    originator: z.ZodOptional<z.ZodObject<{
        accountId: z.ZodOptional<z.ZodNumber>;
        organizationUid: z.ZodOptional<z.ZodString>;
    }, "strict", z.ZodTypeAny, {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    }, {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    }>>;
}, "strict", z.ZodTypeAny, {
    type: "Device.CreateDevice";
    uid: string;
    applicationType: "default" | "sssp" | "tizen" | "webos" | "android" | "chrome" | "brightsign" | "linux" | "windows" | "chromeos";
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    identityHash?: string | undefined;
    originator?: {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    } | undefined;
}, {
    type: "Device.CreateDevice";
    uid: string;
    applicationType: "default" | "sssp" | "tizen" | "webos" | "android" | "chrome" | "brightsign" | "linux" | "windows" | "chromeos";
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    identityHash?: string | undefined;
    originator?: {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    } | undefined;
}>>;
export declare class DeviceIsAlreadyCreatedError {
    readonly duid: DeviceUid;
    static type: 'Device.DeviceIsAlreadyCreatedError';
    readonly type: typeof DeviceIsAlreadyCreatedError.type;
    constructor(duid: DeviceUid);
}
export type UpdateDeviceOrganizationType = 'Device.UpdateDeviceOrganization';
export declare const UpdateDeviceOrganization: UpdateDeviceOrganizationType;
export interface UpdateDeviceOrganization extends ISetDeviceProperty<UpdateDeviceOrganizationType> {
    organizationUid: string;
}
export declare const UpdateDeviceInfo = "Device.UpdateDeviceInfo";
export declare const UpdateDeviceInfoSchema: z.ZodObject<{
    context: z.ZodOptional<z.ZodObject<{
        whiteLabelSettingsUid: z.ZodOptional<z.ZodUnion<[z.ZodString, z.ZodString]>>;
        auth0Enabled: z.ZodOptional<z.ZodBoolean>;
    }, "strict", z.ZodTypeAny, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }>>;
    type: z.ZodLiteral<"Device.UpdateDeviceInfo">;
    deviceUid: z.ZodString;
    model: z.ZodEffects<z.ZodUnion<[z.ZodOptional<z.ZodNullable<z.ZodString>>, z.ZodObject<{
        value: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        value?: string | undefined;
    }, {
        value?: string | undefined;
    }>]>, string | undefined, string | {
        value?: string | undefined;
    } | null | undefined>;
    serialNumber: z.ZodEffects<z.ZodUnion<[z.ZodOptional<z.ZodNullable<z.ZodString>>, z.ZodObject<{
        value: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        value?: string | undefined;
    }, {
        value?: string | undefined;
    }>]>, string | undefined, string | {
        value?: string | undefined;
    } | null | undefined>;
    brand: z.ZodEffects<z.ZodUnion<[z.ZodOptional<z.ZodNullable<z.ZodString>>, z.ZodObject<{
        value: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        value?: string | undefined;
    }, {
        value?: string | undefined;
    }>]>, string | undefined, string | {
        value?: string | undefined;
    } | null | undefined>;
    osVersion: z.ZodEffects<z.ZodUnion<[z.ZodOptional<z.ZodNullable<z.ZodString>>, z.ZodObject<{
        value: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        value?: string | undefined;
    }, {
        value?: string | undefined;
    }>]>, string | undefined, string | {
        value?: string | undefined;
    } | null | undefined>;
}, "strict", z.ZodTypeAny, {
    type: "Device.UpdateDeviceInfo";
    deviceUid: string;
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    model?: string | undefined;
    serialNumber?: string | undefined;
    brand?: string | undefined;
    osVersion?: string | undefined;
}, {
    type: "Device.UpdateDeviceInfo";
    deviceUid: string;
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    model?: string | {
        value?: string | undefined;
    } | null | undefined;
    serialNumber?: string | {
        value?: string | undefined;
    } | null | undefined;
    brand?: string | {
        value?: string | undefined;
    } | null | undefined;
    osVersion?: string | {
        value?: string | undefined;
    } | null | undefined;
}>;
export type UpdateDeviceInfo = z.infer<typeof UpdateDeviceInfoSchema>;
export declare const UpdateDeviceInfoFactory: (data: {
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    brand?: string | {
        value?: string | undefined;
    } | null | undefined;
    model?: string | {
        value?: string | undefined;
    } | null | undefined;
    serialNumber?: string | {
        value?: string | undefined;
    } | null | undefined;
    osVersion?: string | {
        value?: string | undefined;
    } | null | undefined;
} & {
    deviceUid: string;
} & Partial<import("../types").ObjectRecursiveIntersection<{
    type: "Device.UpdateDeviceInfo";
    deviceUid: string;
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    model?: string | undefined;
    serialNumber?: string | undefined;
    brand?: string | undefined;
    osVersion?: string | undefined;
}, {}, Function>>) => z.TypeOf<z.ZodObject<{
    context: z.ZodOptional<z.ZodObject<{
        whiteLabelSettingsUid: z.ZodOptional<z.ZodUnion<[z.ZodString, z.ZodString]>>;
        auth0Enabled: z.ZodOptional<z.ZodBoolean>;
    }, "strict", z.ZodTypeAny, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }>>;
    type: z.ZodLiteral<"Device.UpdateDeviceInfo">;
    deviceUid: z.ZodString;
    model: z.ZodEffects<z.ZodUnion<[z.ZodOptional<z.ZodNullable<z.ZodString>>, z.ZodObject<{
        value: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        value?: string | undefined;
    }, {
        value?: string | undefined;
    }>]>, string | undefined, string | {
        value?: string | undefined;
    } | null | undefined>;
    serialNumber: z.ZodEffects<z.ZodUnion<[z.ZodOptional<z.ZodNullable<z.ZodString>>, z.ZodObject<{
        value: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        value?: string | undefined;
    }, {
        value?: string | undefined;
    }>]>, string | undefined, string | {
        value?: string | undefined;
    } | null | undefined>;
    brand: z.ZodEffects<z.ZodUnion<[z.ZodOptional<z.ZodNullable<z.ZodString>>, z.ZodObject<{
        value: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        value?: string | undefined;
    }, {
        value?: string | undefined;
    }>]>, string | undefined, string | {
        value?: string | undefined;
    } | null | undefined>;
    osVersion: z.ZodEffects<z.ZodUnion<[z.ZodOptional<z.ZodNullable<z.ZodString>>, z.ZodObject<{
        value: z.ZodOptional<z.ZodString>;
    }, "strip", z.ZodTypeAny, {
        value?: string | undefined;
    }, {
        value?: string | undefined;
    }>]>, string | undefined, string | {
        value?: string | undefined;
    } | null | undefined>;
}, "strict", z.ZodTypeAny, {
    type: "Device.UpdateDeviceInfo";
    deviceUid: string;
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    model?: string | undefined;
    serialNumber?: string | undefined;
    brand?: string | undefined;
    osVersion?: string | undefined;
}, {
    type: "Device.UpdateDeviceInfo";
    deviceUid: string;
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    model?: string | {
        value?: string | undefined;
    } | null | undefined;
    serialNumber?: string | {
        value?: string | undefined;
    } | null | undefined;
    brand?: string | {
        value?: string | undefined;
    } | null | undefined;
    osVersion?: string | {
        value?: string | undefined;
    } | null | undefined;
}>>;
export type ISpecs = IAndroidSpecs;
/**
 * Android specific device info specs
 * Example values in comments
 */
export interface IAndroidSpecs {
    build: {
        version: {
            release: string;
            sdk: number;
            codename: string;
        };
        product: {
            brand: string;
            model: string;
            supportedAbis: string[];
            manufacturer?: string;
            device?: string;
            board?: string;
        };
    };
    signatures: {
        framework: {
            sha256: string;
        };
    };
}
export type ChangeDeviceSubscriptionType = 'Device.ChangeDeviceSubscriptionType';
export declare const ChangeDeviceSubscription: ChangeDeviceSubscriptionType;
export interface ChangeDeviceSubscription extends IOriginatorAwareCommand<ChangeDeviceSubscriptionType> {
    deviceUid: DeviceUid;
    subscriptionType: SubscriptionType;
}
export declare const ApproveDevice = "Device.ApproveDevice";
export interface ApproveDevice extends IOriginatorAwareCommand<typeof ApproveDevice> {
    deviceIdentityHash: string;
}
export declare const BanDevice = "Device.BanDevice";
export interface BanDevice extends IOriginatorAwareCommand<typeof BanDevice> {
    deviceIdentityHash: string;
}
export type SyncDeviceOrganizationSubscriptionsType = 'Device.SyncDeviceOrganizationSubscriptions';
export declare const SyncDeviceOrganizationSubscriptions: SyncDeviceOrganizationSubscriptionsType;
export interface SyncDeviceOrganizationSubscriptions extends IBaseCommand<SyncDeviceOrganizationSubscriptionsType> {
}
export declare const UpdateDevice = "Device.UpdateDevice";
export declare const UpdateDeviceSchema: z.ZodObject<{
    context: z.ZodOptional<z.ZodObject<{
        whiteLabelSettingsUid: z.ZodOptional<z.ZodUnion<[z.ZodString, z.ZodString]>>;
        auth0Enabled: z.ZodOptional<z.ZodBoolean>;
    }, "strict", z.ZodTypeAny, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }>>;
    originator: z.ZodObject<{
        accountId: z.ZodOptional<z.ZodNumber>;
        organizationUid: z.ZodOptional<z.ZodString>;
    }, "strict", z.ZodTypeAny, {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    }, {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    }>;
    type: z.ZodLiteral<"Device.UpdateDevice">;
    deviceIdentityHash: z.ZodString;
    sonyCloudDeviceId: z.ZodNullable<z.ZodOptional<z.ZodString>>;
}, "strict", z.ZodTypeAny, {
    type: "Device.UpdateDevice";
    originator: {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    };
    deviceIdentityHash: string;
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    sonyCloudDeviceId?: string | null | undefined;
}, {
    type: "Device.UpdateDevice";
    originator: {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    };
    deviceIdentityHash: string;
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    sonyCloudDeviceId?: string | null | undefined;
}>;
export type UpdateDevice = z.infer<typeof UpdateDeviceSchema>;
export declare const UpdateDeviceFactory: (data: {
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    sonyCloudDeviceId?: string | null | undefined;
} & {
    originator: {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    };
    deviceIdentityHash: string;
} & Partial<import("../types").ObjectRecursiveIntersection<{
    type: "Device.UpdateDevice";
    originator: {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    };
    deviceIdentityHash: string;
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    sonyCloudDeviceId?: string | null | undefined;
}, {}, Function>>) => z.TypeOf<z.ZodObject<{
    context: z.ZodOptional<z.ZodObject<{
        whiteLabelSettingsUid: z.ZodOptional<z.ZodUnion<[z.ZodString, z.ZodString]>>;
        auth0Enabled: z.ZodOptional<z.ZodBoolean>;
    }, "strict", z.ZodTypeAny, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }, {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    }>>;
    originator: z.ZodObject<{
        accountId: z.ZodOptional<z.ZodNumber>;
        organizationUid: z.ZodOptional<z.ZodString>;
    }, "strict", z.ZodTypeAny, {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    }, {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    }>;
    type: z.ZodLiteral<"Device.UpdateDevice">;
    deviceIdentityHash: z.ZodString;
    sonyCloudDeviceId: z.ZodNullable<z.ZodOptional<z.ZodString>>;
}, "strict", z.ZodTypeAny, {
    type: "Device.UpdateDevice";
    originator: {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    };
    deviceIdentityHash: string;
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    sonyCloudDeviceId?: string | null | undefined;
}, {
    type: "Device.UpdateDevice";
    originator: {
        accountId?: number | undefined;
        organizationUid?: string | undefined;
    };
    deviceIdentityHash: string;
    context?: {
        whiteLabelSettingsUid?: string | undefined;
        auth0Enabled?: boolean | undefined;
    } | undefined;
    sonyCloudDeviceId?: string | null | undefined;
}>>;
export {};
