"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContextAwareCommandSchema = void 0;
const zod_1 = require("zod");
const WhiteLabelSettingsUIDSchema_1 = require("../WhiteLabelSettings/WhiteLabelSettingsUIDSchema");
/**
 * Adds context about white label settings
 *
 * When sending certain commands, it's necessary to know the context where the command was produced.
 * It's important to know if the environment that produced that command was a white-labeled environment.
 * We can then use this information to produce events that will reflect that information.
 *
 * For example, if a command came from a white-labeled UI and it produces an event that triggers
 * sending an email to the user, it's important to know that the UI was white-labeled so that
 * the email is also white-labeled the same way.
 */
const WhiteLabelAwareContextSchema = zod_1.z.strictObject({
    whiteLabelSettingsUid: WhiteLabelSettingsUIDSchema_1.WhiteLabelSettingsUIDSchema.optional(),
});
/**
 * Adds context about Auth0
 *
 * We currently support signup/login via both Auth0 and our own custom system.
 * Box is responsible for selecting, which one to use.
 * Depending which one is selected, some commands will have different behavior.
 *
 * For example, if Box sends a command to invite a user and Auth0 is enabled, the email will be sent via Auth0.
 * If Auth0 is disabled, the email will be sent via our own internal service.
 */
const Auth0ContextSchema = zod_1.z.object({
    auth0Enabled: zod_1.z.boolean().optional(),
});
const ContextSchema = zod_1.z.strictObject({}).extend(WhiteLabelAwareContextSchema.shape).extend(Auth0ContextSchema.shape);
/**
 * Adds context to a command
 *
 * Context is information about the environment where the command was produced.
 * It serves as metadata that can be reflected in the produced events.
 */
exports.ContextAwareCommandSchema = zod_1.z.strictObject({
    context: ContextSchema.optional(),
});
//# sourceMappingURL=context.js.map