"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getOriginFromRequestHeaders = exports.getHostnameFromRequestHeaders = void 0;
const url_1 = require("url");
/**
 * Function for obtaining hostname (domain) from request headers.
 * It uses various HTTP headers because of more contexts of usage (production (CloudFront), pre-production (CloudFront), localhost).
 *
 * Used headers:
 * - x-forwarded-host (i.e. 'box.signageos.io', 'box.signageos.io:8080')
 * - origin (i.e. 'https://box.signageos.io', 'https://box.signageos.io:8080')
 * - referer (i.e. 'https://box.signageos.io', 'https://box.signageos.io:8080')
 * - host (i.e. 'box.signageos.io', 'box.signageos.io:8080')
 *
 * @param headers
 */
const getHostnameFromRequestHeaders = (headers) => {
    const origin = (0, exports.getOriginFromRequestHeaders)(headers);
    if (!origin) {
        return undefined;
    }
    const url = new url_1.URL(origin);
    return url.hostname;
};
exports.getHostnameFromRequestHeaders = getHostnameFromRequestHeaders;
const HEADER_X_FORWARDED_HOST = 'x-forwarded-host';
const HEADER_X_FORWARDED_PORT = 'x-forwarded-port';
const HEADER_X_FORWARDED_PROTO = 'x-forwarded-proto';
const HEADER_ORIGIN = 'origin';
const HEADER_REFERER = 'referer';
const HEADER_VIA = 'via';
/** This string value contains the via header when it went through AWS CloudFront */
const VIA_CLOUDFRONT_VALUE = '(CloudFront)';
/**
 * Function for obtaining origin (schema + '://' + domain + ':' + port) from request headers.
 * It uses various HTTP headers because of more contexts of usage (production (CloudFront), pre-production (CloudFront), localhost).
 *
 * Used headers:
 * - x-forwarded-host (i.e. 'box.signageos.io', 'box.signageos.io:8080')
 * - origin (i.e. 'https://box.signageos.io', 'https://box.signageos.io:8080')
 * - referer (i.e. 'https://box.signageos.io', 'https://box.signageos.io:8080')
 * - host (i.e. 'box.signageos.io', 'box.signageos.io:8080')
 *
 * @param headers
 */
const getOriginFromRequestHeaders = (headers) => {
    var _a;
    let protocol = headers[HEADER_X_FORWARDED_PROTO] ? `${headers[HEADER_X_FORWARDED_PROTO]}:` : 'http:';
    let port = headers[HEADER_X_FORWARDED_PORT];
    // TODO: The determining that protocol is https when it goes through CloudFront is temporary solution
    // until the CloudFront will be configured to send the x-forwarded-proto header.
    if ((_a = headers[HEADER_VIA]) === null || _a === void 0 ? void 0 : _a.includes(VIA_CLOUDFRONT_VALUE)) {
        protocol = 'https:';
        port = '443';
    }
    if (!port) {
        port = protocol === 'http:' ? '80' : '443';
    }
    const xForwardedHostHeader = headers[HEADER_X_FORWARDED_HOST];
    if (typeof xForwardedHostHeader !== 'undefined' && !Array.isArray(xForwardedHostHeader)) {
        let portPart = '';
        if ((protocol === 'https:' && port !== '443') || (protocol === 'http:' && port !== '80')) {
            portPart = `:${port}`;
        }
        return `${protocol}//${xForwardedHostHeader}${portPart}`;
    }
    // There is an edge case, when the request comes from file:// URL
    // The ws server library produces a request where the origin header is "null" as a string.
    // Probably some bug in the library, but we need to handle it
    if (headers[HEADER_ORIGIN] === 'null') {
        delete headers[HEADER_ORIGIN];
    }
    const originHeader = headers[HEADER_ORIGIN];
    const originUrl = originHeader && !Array.isArray(originHeader) ? new url_1.URL(originHeader) : undefined;
    if (originUrl === null || originUrl === void 0 ? void 0 : originUrl.origin) {
        return originUrl.origin;
    }
    const refererHeader = headers[HEADER_REFERER];
    const refererUrl = refererHeader ? new url_1.URL(refererHeader) : undefined;
    if (refererUrl === null || refererUrl === void 0 ? void 0 : refererUrl.origin) {
        return refererUrl.origin;
    }
    const hostHeader = headers.host;
    if (typeof hostHeader !== 'undefined') {
        return protocol + '//' + hostHeader;
    }
    return undefined;
};
exports.getOriginFromRequestHeaders = getOriginFromRequestHeaders;
//# sourceMappingURL=http.js.map