"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.envVarsServiceConfigSchema = exports.packageConfigSchema = exports.jsonStringSchema = exports.boolSchema = exports.intSchema = void 0;
const zod_1 = require("zod");
const parsers_1 = require("../Parsers/parsers");
const lodash_1 = require("lodash");
/**
 * Schema for integer string
 *
 * Parses a string into an integer or throws an error if the string is not a valid integer.
 */
exports.intSchema = zod_1.z
    .string()
    .transform((value) => parseInt(value))
    .refine((value) => (0, lodash_1.isNumber)(value) && !(0, lodash_1.isNaN)(value), 'must be a number');
/**
 * Schema for boolean string
 *
 * Parses a string into a boolean or throws an error if the string is not a valid boolean.
 * Accepted values are 'true', 'false', '1', '0'
 */
exports.boolSchema = zod_1.z.string().optional().transform(parsers_1.parseBoolean);
/**
 * Schema for JSON string
 *
 * Parses a string into a JSON object or throws an error if the string is not a valid JSON
 */
exports.jsonStringSchema = zod_1.z.string().transform((str, ctx) => {
    try {
        return JSON.parse(str);
    }
    catch (error) {
        ctx.addIssue({ code: 'custom', message: error.message });
        return zod_1.z.NEVER;
    }
});
/**
 * Schema for package.json
 */
exports.packageConfigSchema = zod_1.z.object({
    name: zod_1.z.string(),
    version: zod_1.z.string(),
    dependencies: zod_1.z.record(zod_1.z.string()).default({}),
    devDependencies: zod_1.z.record(zod_1.z.string()).default({}),
    peerDependencies: zod_1.z.record(zod_1.z.string()).default({}),
    optionalPeerDependencies: zod_1.z.record(zod_1.z.string()).default({}),
    gitHead: zod_1.z.string().optional(),
});
/**
 * Expected environment variables for services
 */
exports.envVarsServiceConfigSchema = zod_1.z.object({
    PORT: exports.intSchema.default('8080'),
    MONITORING_PORT: exports.intSchema.default('9999'),
});
//# sourceMappingURL=validation.js.map