export type QueueType = 'quorum' | 'classic';
type IAutoDeleteDeadLetterOptions = {
    /**
     * Queue is auto deleted when consumption is canceled
     * For sure, queue is set as x-expires=
     * This is proprietary option, not amqp or rabbit native
     */
    autoDeleteDeadLetter: true;
    /** Imply true because if consumption is not exclusive it cannot detect no consumer and auto rejecting messages */
    exclusiveConsumption: true;
    /**
     * Cannot be autoDelete because is deleted by application after consumption is canceled after messages are dead-lettered
     * But all bindings should be persistent=false to make it work properly
     */
    autoDelete: false;
    /** Cannot be exclusive because is deleted by application after consumption is canceled after messages are dead-lettered */
    exclusive: false;
    /** Has to have at least deadLetterExchange because otherwise, it can be used just standard amqp autoDelete */
    deadLetterExchange: string;
    deadLetterRoutingKey: string;
    /**
     * Has to be set to allow delete queue even when amqp connection is closed before application can dead-letter all messages.
     * Messages have to have shorter expiration than queueExpiresMs to make it work properly
     */
    queueExpiresMs: number;
};
type IBasicConsumeOptions = {
    /** Auto delete queue when number of consumers decreased to zero */
    autoDelete?: boolean;
    /** Only one consumer can be for queue, after consumption canceled, queue is deleted */
    exclusive?: boolean;
    /** Queue is deleted when rabbitmq is restarted (keeps schema in memory only) */
    durable?: boolean;
    exclusiveConsumption?: boolean;
    deadLetterExchange?: string;
    deadLetterRoutingKey?: string;
    redeliverDelayMs?: number;
    suppressFirstError?: boolean;
    /**
     * If flag is set, the message will be rejected with requeue=false in case
     * the message was redelivered already and now the error was thrown
     */
    deadLetterIfRedeliveredAndErred?: boolean;
    /**
     * It will immediately dead-letter the message if the error is thrown during the consumption
     * Unlike deadLetterIfRedeliveredAndErred, it does not care about redelivered flag and dead-letters immediately
     */
    deadLetterIfErred?: boolean;
    /** If respond is set, the consumption will reply response message if replyTo parameter is set in coming message. Default false */
    respond?: boolean;
    /** Define how long the queue will exists without consumer until is automatically deleted (RabbitMQ feature only) */
    queueExpiresMs?: number;
    /**
     * Define how long can be the messages in the queue until they are automatically rejected (dead-lettered).
     * Internally used header 'x-message-ttl'.
     * It can be set also as 'expiration' attribute of every message. See IMessageOptions.
     */
    messageExpiresMs?: number;
    /**
     * Defines if a queue is allowed to be consumed by only one single consumer at the same time (RabbitMQ 3.8+ feature only)
     * https://www.rabbitmq.com/consumers.html#single-active-consumer
     */
    singleActiveConsumer?: boolean;
    /**
     * Detect whether the queue is the quorum type or classic.
     * Quorum is better option for consistency of message of durable queue.
     * Default is classic.
     */
    queueType?: QueueType;
    /**
     * When is true, all messages are acknowledged immediately after client populate them.
     * There is not need to call ack() method on message.
     */
    noAck?: boolean;
    /**
     * Limit the number of messages in the queue.
     * When the limit is reached, the oldest messages are handled by the overflow policy.
     * If overflow is not set, the default policy is to drop the oldest messages.
     */
    maxLength?: number;
    /**
     * Internally used header 'x-overflow'.
     * Sets the strategy for handling messages when the queue reaches its maximum length @see `maxLength`.
     * The default policy is to drop the oldest messages.
     * The other options are:
     * - `reject-publish`: Reject the message and do not publish it to the queue.
     * - `reject-publish-dlx`: Reject the message and publish it to the dead-letter exchange.
     * - `drop-head`: Drop the oldest message in the queue.
     */
    overflow?: 'reject-publish' | 'reject-publish-dlx' | 'drop-head';
};
type IQuorumOptions = IBasicConsumeOptions & {
    durable: true;
    queueType: 'quorum';
};
type IConsumeOptions = IBasicConsumeOptions | IQuorumOptions | (IBasicConsumeOptions & IAutoDeleteDeadLetterOptions) | (IQuorumOptions & IAutoDeleteDeadLetterOptions);
export default IConsumeOptions;
