
class ServiceConfig {
  [xml] $xml = (New-Object System.Xml.XmlDocument)
  [System.Xml.XmlElement] $service

  ServiceConfig() {
    $this.service = $this.xml.CreateElement("service")
    $this.xml.AppendChild($this.service) | Out-Null
  }

  [void] Save([string] $Path) {
    $directory = Split-Path -Path $Path
    New-Item -Path $directory -ItemType Directory | Out-Null
    $this.xml.Save($Path)
  }

  [void] SetServiceElement([System.Xml.XmlElement] $Element) {
    $this.service.AppendChild($Element) | Out-Null
  }

  [void] SetServiceElement([string] $ElementName, [string] $ElementValue) {
    $this.SetElementToParent($ElementName, $ElementValue, $this.service)
  }

  [void] SetElementToParent(
    [string] $ElementName,
    [string] $ElementValue,
    [System.Xml.XmlElement] $Parent
  ) {
    $element = $this.xml.CreateElement($ElementName)
    $element.InnerText = $ElementValue
    $Parent.AppendChild($element) | Out-Null
  }

  [System.Xml.XmlElement] GetElementWithAttributes([string] $ElementName, [object[]] $Attributes) {
    $element = $this.xml.CreateElement($ElementName)
    Foreach ($Attribute in $Attributes) {
      $element.SetAttribute($Attribute.Name, $Attribute.Value) | Out-Null
    }
    return $element
  }

  [void] SetServiceElementWithAttributes([string] $ElementName, [object[]] $Attributes) {
    $element = $this.GetElementWithAttributes($ElementName, $Attributes)
    $this.SetServiceElement($element)
  }

  [void] SetId([string] $Id) {
    $this.SetServiceElement("id", $Id)
  }

  [void] SetName([string] $Name) {
    $this.SetServiceElement("name", $Name)
  }

  [void] SetDescription([string] $Description) {
    $this.SetServiceElement("description", $Description)
  }

  [void] SetExecutable([string] $Executable) {
    $this.SetServiceElement("executable", $Executable)
  }

  [void] SetArgument([string] $Argument) {
    $this.SetServiceElement("argument", $Argument)
  }

  [void] SetLog([string] $Mode) {
    $this.SetServiceElementWithAttributes(
      "log",
      @(@{Name="mode";Value=$Mode})
    )
  }

  [void] SetLogModeRollBySizeTime(
    [string] $Pattern,
    [string] $AutoRollAtTime,
    [string] $SizeThreshold,
    [int] $KeepFiles
  ) {
    $log = $this.GetElementWithAttributes(
      "log",
      @(@{Name="mode";Value="roll-by-size-time"})
    )

    $this.SetElementToParent("sizeThreshold", $SizeThreshold, $log)
    $this.SetElementToParent("keepFiles", $KeepFiles, $log)
    $this.SetElementToParent("pattern", $Pattern, $log)
    $this.SetElementToParent("autoRollAtTime", $AutoRollAtTime, $log)

    $this.SetServiceElement($log)
  }

  [void] SetOnFailure([string] $Action, [string] $Delay) {
    $this.SetServiceElementWithAttributes(
      "onfailure",
      @(
        @{Name="action";Value=$Action};
        @{Name="delay";Value=$Delay};
      )
    )
  }

  [void] SetOnFailure([string] $Action) {
    $this.SetServiceElementWithAttributes(
      "onfailure",
      @(@{Name="action";Value=$Action})
    )
  }

  [void] SetStopTimeout([string] $StopTimeout) {
    $this.SetServiceElement("stoptimeout", $StopTimeout)
  }

  [void] SetWorkingDirectory([string] $WorkingDirectory) {
    $this.SetServiceElement("workingdirectory", $WorkingDirectory)
  }

  [void] SetEnvironmentalVariable([string] $Name, [string] $Value) {
    $this.SetServiceElementWithAttributes(
      "env",
      @(
        @{Name="name";Value=$Name};
        @{Name="value";Value=$Value};
      )
    )
  }

  [void] SetDependency([string] $Name) {
    $this.SetServiceElement("depend", $Name)
  }
}

function Get-ServiceId([string] $Name) {
  return ($Name -replace "\s+", "").ToLower()
}

function New-ServiceConfig(
  [string] $Id,
  [string] $Name,
  [string] $Description
  ) {
  [ServiceConfig] $service = [ServiceConfig]::new()

  $service.SetId($Id + ".exe")
  $service.SetName($Name)
  $service.SetDescription($Description)

  return $service
}

function New-NodeServiceConfig(
  [string] $Id,
  [string] $Name,
  [string] $Description,
  [string] $NodeExecutablePath,
  [int] $MaxAvailableMemory,
  [string] $ServiceExecutableFilePath
  ) {
  [ServiceConfig] $service = New-ServiceConfig $Id $Name $Description

  $service.SetExecutable($NodeExecutablePath)
  $service.SetArgument("--max_old_space_size=$MaxAvailableMemory")
  $service.SetArgument($ServiceExecutableFilePath)

  return $service
}

Export-ModuleMember -Function New-ServiceConfig, New-NodeServiceConfig, Get-ServiceId
