$IsEmptyPasswordFromInstaller = ($null -eq $env:IsEmptyPasswordFromCLIArg) -and ($null -eq $env:KioskUserPassword)
if (($IsEmptyPasswordFromInstaller -eq $true) -or ($env:IsEmptyPasswordFromCLIArg -eq $true)) {
  $KioskPassword = ""
} else {
  $KioskPassword = $env:KioskUserPassword
}

Import-Module $PSScriptRoot\AutoAdminLogon\AutoAdminLogon.psm1 -Force
Import-Module $PSScriptRoot\Get-PlainTextFromSecureString.psm1 -Force

function Read-PasswordWithConfirm {
  param (
      [Parameter(Mandatory=$true)]
      [string]
      $Help,

      [Parameter(Mandatory=$true)]
      [string]
      $ConfirmHelp
  )
  $TriesCounter = 0
  $TriesLimit = 3

  do {
    $securePassword = Read-Host -Prompt $Help -AsSecureString
    $confirmSecurePassword = Read-Host -Prompt $ConfirmHelp -AsSecureString
    $securePasswordPlainText = Get-PlainTextFromSecureString -SecureString $securePassword
    $confirmSecurePasswordPlainText = Get-PlainTextFromSecureString -SecureString $confirmSecurePassword

    $arePasswordsMatching = $securePasswordPlainText.Equals($confirmSecurePasswordPlainText)
    $validPassword = $arePasswordsMatching
    if (-not $arePasswordsMatching) {
      Write-Host "Error: Passwords don't match. Try again." -ForegroundColor Red
    }
    $TriesCounter++
  } while ((-not $validPassword) -and ($TriesCounter -lt $TriesLimit))

  if ((-not $validPassword) -and ($TriesCounter -ge $TriesLimit)) {
    throw "Password retry limit $TriesLimit exhausted. Please start the installation process from beggining."
  }

  return $securePassword
}

$username = $env:KioskUsername
$securePassword = New-Object System.Security.SecureString

if ($KioskPassword -eq $null) {
  $securePassword = Read-PasswordWithConfirm -Help "Enter Kiosk user password" -ConfirmHelp "Confirm Kiosk user password"
} elseif ("" -ne $KioskPassword) {
  $securePassword = ConvertTo-SecureString $KioskPassword -AsPlainText -Force
}

$plainTextPassword = Get-PlainTextFromSecureString $securePassword
Set-Item -Path "env:KioskUserPassword" -Value $plainTextPassword
[System.Environment]::SetEnvironmentVariable("KioskUserPassword", $plainTextPassword, [System.EnvironmentVariableTarget]::User)

Write-Host "Creating local user: $username"
$user = New-LocalUser -Name $username -Password $securePassword -UserMayNotChangePassword -AccountNeverExpires -PasswordNeverExpires
$usersLocalGroupSID = New-Object System.Security.Principal.SecurityIdentifier("S-1-5-32-545")
Add-LocalGroupMember -SID $usersLocalGroupSID -Member $user

Write-Host "Local user: $username has been created."

Write-Host "Configuring automatic login"

Set-AutoAdminLogon -Username $username -Password $securePassword
